package cn.net.wanmo.common.util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.*;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.lang3.StringEscapeUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Map;
import java.util.Objects;
import java.util.TimeZone;

/**
 * Jackson可以轻松的将Java对象转换成json对象和xml文档，同样也可以将json、xml转换成Java对象..
 * 如果需要转换xml，那么还需要stax2-api.jarb<br>
 * <p>
 * Jackson提供了一系列注解，方便对JSON序列化和反序列化进行控制，下面介绍一些常用的注解。
 * // @ JsonIgnore 此注解用于属性上，作用是进行JSON操作时忽略该属性。
 * // @ JsonFormat 此注解用于属性上，作用是把Date类型直接转化为想要的格式，如@JsonFormat(pattern = "yyyy-MM-dd HH-mm-ss")。
 * // @ JsonProperty 此注解用于属性上，作用是把该属性的名称序列化为另外一个名称，如把trueName属性序列化为name，@JsonProperty("name")。
 * // http://blog.csdn.net/jesse621/article/details/22698565
 */
public class JacksonMapper extends ObjectMapper {
    private static Logger logger = LoggerFactory.getLogger(JacksonMapper.class);
    private static JacksonMapper mapper = null;

    public JacksonMapper() {
        this(JsonInclude.Include.NON_EMPTY);
    }

    public JacksonMapper(JsonInclude.Include include) {
        // 设置输出时包含属性的风格
        if (Objects.nonNull(include)) {
            this.setSerializationInclusion(include);
        }

        // 允许单引号、允许不带引号的字段名称
        this.enableSimple();
        // 设置输入时忽略JSON字符串中存在但java对象实际没有的属性
        this.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        // 空值处理为空串
        this.getSerializerProvider().setNullValueSerializer(new JsonSerializer<Object>() {
            @Override
            public void serialize(Object value, JsonGenerator jgen,
                                  SerializerProvider provider) throws IOException,
                    JsonProcessingException {
                jgen.writeString("");
            }
        });
        // 进行HTML解码。
        this.registerModule(new SimpleModule().addSerializer(String.class, new JsonSerializer<String>() {
            @Override
            public void serialize(String value, JsonGenerator jgen,
                                  SerializerProvider provider) throws IOException,
                    JsonProcessingException {
                jgen.writeString(StringEscapeUtils.unescapeHtml4(value));
            }
        }));
        // 设置时区
        this.setTimeZone(TimeZone.getDefault());//getTimeZone("GMT+8:00")
    }

    /**
     * 允许单引号
     * 允许不带引号的字段名称
     */
    public JacksonMapper enableSimple() {
        this.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
        this.configure(JsonParser.Feature.ALLOW_UNQUOTED_FIELD_NAMES, true);
        return this;
    }

    /**
     * 返回mapper，忽略未知参数、使用long解析时间
     *
     * @return ObjectMapper
     */
    public static JacksonMapper getMapper() {
        if (mapper == null) {
            mapper = new JacksonMapper().enableSimple();
        }
        return mapper;
    }

    /**
     * 创建只输出初始值被改变的属性到Json字符串的Mapper, 最节约的存储方式，建议在内部接口中使用。
     */
    public static JacksonMapper nonDefaultMapper() {
        if (mapper == null) {
            mapper = new JacksonMapper(JsonInclude.Include.NON_DEFAULT);
        }
        return mapper;
    }

    /**
     * java对象转换成JSON字符串<br>
     * java对象、List、Map
     */
    public static String toJson(Object obj) {
        String json = "";
        try {
            json = getMapper().writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            logger.warn("write to json string error:" + obj, e);
        }
        return json;
    }

    /**
     * 读取JSON数据: 从 byte[]
     *
     * @param json
     * @param clazz
     * @return 对象
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public static <T> T fromJson(byte[] json, Class<T> clazz) {
        try {
            return getMapper().readValue(json, clazz);
        } catch (IOException e) {
            logger.warn("parse json string error:", e);
        }
        return null;
    }

    /**
     * 读取JSON数据: 从 InputStream
     *
     * @param json
     * @param valueType
     * @return 泛型对象
     */
    public static <T> T fromJson(InputStream json, Class<T> valueType) {
        try {
            return getMapper().readValue(json, valueType);
        } catch (IOException e) {
            logger.warn("parse json string error:", e);
        }
        return null;
    }

    /**
     * 读取JSON数据: 从字符串
     *
     * @param json
     * @param valueType
     * @return 泛型对象
     */
    public static <T> T fromJson(String json, Class<T> valueType) {
        try {
            return getMapper().readValue(json, valueType);
        } catch (IOException e) {
            logger.warn("parse json string error:" + json, e);
        }
        return null;
    }

    /**
     * 读取JSON数据: 从文件
     *
     * @param json
     * @param valueType
     * @return 泛型对象
     */
    public static <T> T fromJson(File json, Class<T> valueType) {
        try {
            return getMapper().readValue(json, valueType);
        } catch (IOException e) {
            logger.warn("parse json string error:" + json, e);
        }
        return null;
    }

    /**
     * 该方法有时有问题
     *
     * @param json
     * @param javaType
     * @return 泛型对象
     * @throws Exception
     */
    public static <T> T fromJson(String json, JavaType javaType) throws Exception {
        return getMapper().readValue(json, javaType);
    }

    /**
     * 将 json 转为 List, 该方法有时有问题
     *
     * @param json
     * @param valueType 对象类型
     * @return 泛型对象
     * @throws Exception
     */
    public static <T> T jsonToList(String json, Class<T> valueType) throws Exception {
        JavaType javaType = getCollectionType(ArrayList.class, valueType);
        return mapper.readValue(json, javaType);
    }

    /**
     * 获取泛型的Collection Type
     *
     * @param collectionClass 泛型的Collection
     * @param elementClasses  元素类
     * @return JavaType Java类型
     * @since 1.0
     */
    public static JavaType getCollectionType(Class<?> collectionClass, Class<?>... elementClasses) {
//		return mapper.getTypeFactory().constructParametrizedType(collectionClass, collectionClass, elementClasses);
//		TypeFactory.defaultInstance().constructParametrizedType(parametrized, parametersFor, parameterClasses)
        return mapper.getTypeFactory().constructParametricType(collectionClass, elementClasses);
//		mapper.getTypeFactory().constructParametricType(parametrized, parameterClasses)
    }

    /**
     * 写入JSON数据：到 文件
     *
     * @param file
     * @param value
     */
    public static <T> void writeJson(File file, T value) {
        try {
            getMapper().writeValue(file, value);
        } catch (IOException e) {
            logger.warn("write json to file error:", e);
        }
    }


    /**
     * JSON 字符串解析为 JsonNode
     *
     * @param content
     * @return JSON 字符串解析为 JsonNode
     * @throws IOException
     */
    public static JsonNode toNode(String content) throws IOException {
        return getMapper().readTree(content);
    }

    /**
     * 创建 节点 对象
     *
     * @param map
     * @return 节点 对象
     */
    public static ObjectNode createObjectNode(Map<String, String> map) {
        ObjectNode jsonObject = getMapper().createObjectNode();
        for (Map.Entry<String, String> entry : map.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            jsonObject.put(key, value);
        }
        return jsonObject;
    }

    public static void main(String[] args) {
        String json = "{\"access_token\":\"8x_u-fYL2ZrmiitGIQTRxb6UpZr8uAWQHEd-Cp4NQB1_P_nrw3kLyDAklwbWK3W7SIgHcc1_DZ2paxz8DR34I3UYy66IcPDlqOSpNOpjN1gzvy4nIeDxF3_W8Zq6DTZ_IBOaAIAEXK\",\"expires_in\":7200}";
        Map map = fromJson(json, Map.class);
        System.out.println(map);

//		AccessToken accessToken = AccessTokenUtil.getAccessToken("wxa6ca97be37890e52", "e7c0b3f8807771a41ca75652169b9aa3");
//		System.out.println(BeanUtil.print(accessToken));
    }

}
