package cn.net.wanmo.common.util;

import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;

import java.lang.management.ManagementFactory;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Date;

/**
 * 日期工具类
 */
public class DateUtil extends DateUtils {
    public static String YYYY = "yyyy";
    public static String YYYY_MM = "yyyy-MM";
    public static String YYYY_MM_DD = "yyyy-MM-dd";
    public static String YYYYMMDDHHMMSS = "yyyyMMddHHmmss";
    public static String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    private static String[] parsePatterns = {"yyyy-MM-dd", "yyyy-MM-dd HH:mm:ss", "yyyy-MM-dd HH:mm", "yyyy-MM",
            "yyyy/MM/dd", "yyyy/MM/dd HH:mm:ss", "yyyy/MM/dd HH:mm", "yyyy/MM", "yyyy.MM.dd", "yyyy.MM.dd HH:mm:ss",
            "yyyy.MM.dd HH:mm", "yyyy.MM", "yyyy-MM-dd HH:mm:ss.s", "yyyy-MM-dd HH:mm:ss.ss",
            "yyyy-MM-dd HH:mm:ss.sss"};

    /**
     * 获取当前 Calendar
     *
     * @return 当前 Calendar
     */
    public static Calendar nowCalendar() {
        return Calendar.getInstance();
    }

    /**
     * 获取当前 Date
     *
     * @return 当前 Date
     */
    public static Date nowDate() {
        return nowCalendar().getTime();
    }

    /**
     * 获取当前 Long
     *
     * @return 当前时间的 Long 类型值
     */
    public static long nowLong() {
        return nowDate().getTime();
    }

    /**
     * 当前时间 加 millisecond 毫秒
     *
     * @param millisecond 追加的毫秒
     * @return 追加后的 Date
     */
    public static Date addMilliseconds(int millisecond) {
        return addMilliseconds(nowDate(), millisecond);
    }

    /**
     * 当前时间 加 millisecond 毫秒
     *
     * @param date 日期
     * @param millisecond 追加的毫秒
     * @return 追加后的 Date
     */
    public static Date addMilliseconds(Date date, int millisecond) {
        return add(date, Calendar.MILLISECOND, millisecond);
    }


    /**
     * 当前时间 加 second 秒
     *
     * @param second 追加的秒数
     * @return 追加后的 Date
     */
    public static Date addSeconds(int second) {
        return addSeconds(nowDate(), second);
    }

    /**
     * 当前时间 加 second 秒
     *
     * @param date 日期
     * @param second 追加的秒数
     * @return 追加后的 Date
     */
    public static Date addSeconds(Date date, int second) {
        return add(date, Calendar.SECOND, second);
    }

    /**
     * 当前时间 加 minute 分钟
     *
     * @param minute 追加的分钟
     * @return 追加后的 Date
     */
    public static Date addMinutes(int minute) {
        return addMinutes(nowDate(), minute);
    }

    /**
     * 当前时间 加 minute 秒
     *
     * @param date 日期
     * @param minute 追加的分钟
     * @return 追加后的 Date
     */
    public static Date addMinutes(Date date, int minute) {
        return add(date, Calendar.MINUTE, minute);
    }

    /**
     * 当前时间 加 hour 小时
     *
     * @param hour 追加的小时
     * @return 追加后的 Date
     */
    public static Date addHours(int hour) {
        return addHours(nowDate(), hour);
    }

    /**
     * 当前时间 加 hour 小时
     *
     * @param date 日期
     * @param hour 追加的小时
     * @return 追加后的 Date
     */
    public static Date addHours(Date date, int hour) {
        return add(date, Calendar.HOUR_OF_DAY, hour);
    }

    /**
     * 追加时间
     * @param date 日期
     * @param calendarField 追加类型： 1则代表的是对年份操作，2是对月份操作，3是对星期操作，5是对日期操作，11是对小时操作，12是对分钟操作，13是对秒操作，14是对毫秒操作
     * @param amount 追加数量
     * @return 追加后的日期
     */
    public static Date add(Date date, int calendarField, int amount) {
        if (date == null) {
            throw new IllegalArgumentException("The date must not be null");
        } else {
            Calendar c = Calendar.getInstance();
            c.setTime(date);
            c.add(calendarField, amount);
            return c.getTime();
        }
    }


    /**
     * 获取n天之前的时间
     *
     * @param date 日期
     * @param day  天数
     * @return 日期
     */
    public static Date getDateBefore(Date date, int day) {
        nowCalendar().setTime(date);
        nowCalendar().set(Calendar.DATE, nowCalendar().get(Calendar.DATE) - day);
        return nowCalendar().getTime();
    }

    /**
     * 获取n天之后的时间
     *
     * @param date 日期
     * @param day  天数
     * @return 日期
     */
    public static Date getDateAfter(Date date, int day) {
        nowCalendar().setTime(date);
        nowCalendar().set(Calendar.DATE, nowCalendar().get(Calendar.DATE) + day);
        return nowCalendar().getTime();
    }

    /**
     * 获取n分钟前的时间
     *
     * @param date 日期
     * @param day  天数
     * @return 日期
     */
    public static Date getDateBeforeMinute(Date date, int day) {
        nowCalendar().setTime(date);
        nowCalendar().set(Calendar.MINUTE, nowCalendar().get(Calendar.MINUTE) - day);
        return nowCalendar().getTime();
    }

    /**
     * 获取n分钟后的时间
     *
     * @param date 日期
     * @param day  天数
     * @return 日期
     */
    public static Date getDateAfterMinute(Date date, int day) {
        nowCalendar().setTime(date);
        nowCalendar().set(Calendar.MINUTE, nowCalendar().get(Calendar.MINUTE) + day);
        return nowCalendar().getTime();
    }

    /**
     * 日期字符串解析
     *
     * @param source 日期字符串
     * @return 日期
     */
    public static Date parse(String source) {
        if (source == null) {
            return null;
        }

        try {
            return parseDate(source, parsePatterns);
        } catch (ParseException e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 日期格式化， 默认（yyyy-MM-dd）
     *
     * @param date    日期
     * @param pattern 格式化字符串
     * @return 日期字符串
     */
    public static String format(Date date, String... pattern) {
        if (date == null) {
            return null;
        }

        if (pattern == null || pattern.length == 0) {
            return DateFormatUtils.format(date, "yyyy-MM-dd");
        } else {
            return DateFormatUtils.format(date, pattern[0]);
        }

    }

    /**
     * 得到UTC时间，类型为字符串
     *
     * @param date    日期
     * @param pattern 格式化字符串
     * @return 日期字符串
     */
    public static String formatUTC(Date date, String pattern) {
        return DateFormatUtils.formatUTC(date, pattern);
    }

    /**
     * 日期格式化
     *
     * @param date    日期
     * @param pattern 格式化字符串
     * @return 日期字符串
     */
    public static String formatDate(Date date, String pattern) {
        return format(date, pattern);
    }

    /**
     * 格式化当前日期 （yyyy-MM-dd）
     *
     * @return 当前日期字符串
     */
    public static String formatDay() {
        return formatDay(nowDate());
    }

    /**
     * 格式化日期 （yyyy-MM-dd）
     *
     * @param date 日期
     * @return 日期字符串
     */
    public static String formatDay(Date date) {
        return format(date, "yyyy-MM-dd");
    }

    /**
     * 格式化当前日期 （yyyy-MM-dd HH:mm:ss）
     *
     * @return 日期字符串
     */
    public static String formatDayTime() {
        return formatDayTime(nowDate());
    }

    /**
     * 格式化日期 （yyyy-MM-dd HH:mm:ss）
     *
     * @return 日期字符串
     */
    public static String formatDayTime(Date date) {
        return format(date, "yyyy-MM-dd HH:mm:ss");
    }

    /**
     * 得到当前年份字符串 格式（yyyy）
     */
    public static String getYear() {
        return format(nowDate(), "yyyy");
    }

    /**
     * 得到当前月份字符串 格式（MM）
     */
    public static String getMonth() {
        return format(nowDate(), "MM");
    }

    /**
     * 得到当天字符串 格式（dd）
     */
    public static String getDay() {
        return format(nowDate(), "dd");
    }

    /**
     * 得到当天字符串 格式（HH）
     */
    public static String getHour() {
        return format(nowDate(), "HH");
    }

    /**
     * 得到当前星期字符串 格式（E）星期几
     */
    public static String getWeek() {
        return format(nowDate(), "E");
    }

    /**
     * 获取过去的天数
     *
     * @param date 日期
     * @return 天数 long
     */
    public static long pastDays(Date date) {
        long t = nowLong() - date.getTime();
        return t / (24 * 60 * 60 * 1000);
    }

    /**
     * 获取过去的小时
     *
     * @param date 日期
     * @return 小时 Long
     */
    public static long pastHour(Date date) {
        long t = nowLong() - date.getTime();
        return t / (60 * 60 * 1000);
    }

    /**
     * 获取过去的分钟
     *
     * @param date 日期
     * @return 分钟 long
     */
    public static long pastMinutes(Date date) {
        long t = nowLong() - date.getTime();
        return t / (60 * 1000);
    }

    /**
     * 获取服务器启动时间
     */
    public static Date getServerStartDate() {
        long time = ManagementFactory.getRuntimeMXBean().getStartTime();
        return new Date(time);
    }

    /**
     * 计算相差天数
     */
    public static int differentDaysByMillisecond(Date date1, Date date2) {
        return Math.abs((int) ((date2.getTime() - date1.getTime()) / (1000 * 3600 * 24)));
    }

    /**
     * 计算两个时间差
     */
    public static String getDatePoor(Date endDate, Date nowDate) {
        long nd = 1000 * 24 * 60 * 60;
        long nh = 1000 * 60 * 60;
        long nm = 1000 * 60;
        // long ns = 1000;
        // 获得两个时间的毫秒时间差异
        long diff = endDate.getTime() - nowDate.getTime();
        // 计算差多少天
        long day = diff / nd;
        // 计算差多少小时
        long hour = diff % nd / nh;
        // 计算差多少分钟
        long min = diff % nd % nh / nm;
        // 计算差多少秒//输出结果
        // long sec = diff % nd % nh % nm / ns;
        return day + "天" + hour + "小时" + min + "分钟";
    }
}
