package cn.net.wanmo.common.cert;

import cn.net.wanmo.common.crypto.rsa.RsaUtil;
import cn.net.wanmo.common.util.Exceptions;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509v3CertificateBuilder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * RSA证书工具类
 */
public class RsaCertUtil {
    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    /**
     * 生成 CA 根证书 <br/>
     * CA生成以后把私钥和证书一起以一个key entry的方式存入一个jks文件。 <br/>
     * 生成CA证书与生成普通证书的区别是： <br/>
     * 1，生成CA证书时，issuer和subject一致； <br/>
     * 2，在ContentSigner.build()的时候（签名的时候）使用的是与待签名公钥相应的私钥。
     * @param file 如：resource/atlas-ca.jks
     */
    public static void createRootCA(File file) {
        try {
            final KeyPair keyPair = RsaUtil.getKeyPair();

            String issuer = "C=CN,ST=GuangDong,L=Shenzhen,O=Skybility,OU=Cloudbility,CN=Atlas Personal License CA,E=cwjcsu@126.com";
            String subject = issuer;

            // issuer 与 subject相同的证书就是CA证书
            Certificate cert = generateV3(issuer, subject, BigInteger.ZERO,
                    new Date(System.currentTimeMillis() - 1000 * 60 * 60 * 24),
                    new Date(System.currentTimeMillis() + 1000L * 60 * 60 * 24 * 365 * 32), keyPair.getPublic(), // 待签名的公钥
                    keyPair.getPrivate(), null);

            KeyStore store = KeyStore.getInstance("JKS");
            store.load(null, null);
            store.setKeyEntry("atlas", keyPair.getPrivate(), "atlas".toCharArray(), new Certificate[] { cert });
            cert.verify(keyPair.getPublic());

            if (file.exists() || file.createNewFile()) {
                store.store(new FileOutputStream(file), "atlas".toCharArray());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 生成证书信息
     * @param issuer 颁发者
     * @param subject 使用者
     * @param serial 证书序列号
     * @param notBefore 生效时间
     * @param notAfter 失效时间
     * @param publicKey 公钥
     * @param privKey 私钥
     * @param extensions 其他
     * @return 证书信息
     */
    public static Certificate generateV3(String issuer, String subject,
                                         BigInteger serial, Date notBefore, Date notAfter,
                                         PublicKey publicKey, PrivateKey privKey, List<Extension> extensions) {
        try {
            X509v3CertificateBuilder builder = new JcaX509v3CertificateBuilder(
                    new X500Name(issuer), serial, notBefore, notAfter,
                    new X500Name(subject), publicKey);

            // privKey: 使用自己的私钥进行签名，CA证书
            ContentSigner sigGen = new JcaContentSignerBuilder("SHA1withRSA").setProvider("BC").build(privKey);

            if (extensions == null) {
                extensions = new ArrayList<>();
            }
            for (Extension ext : extensions) {
                builder.addExtension(new ASN1ObjectIdentifier(ext.getOid()), ext.isCritical(), ASN1Primitive.fromByteArray(ext.getValue()));
            }

            X509CertificateHolder holder = builder.build(sigGen);
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            InputStream is1 = new ByteArrayInputStream(holder.toASN1Structure().getEncoded());
            X509Certificate theCert = (X509Certificate) cf.generateCertificate(is1);
            is1.close();
            return theCert;
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 待完善
     */
    public class Extension {
        private String oid;
        private boolean critical;
        private byte[] value;

        public String getOid() {
            return oid;
        }

        public byte[] getValue() {
            return value;
        }

        public boolean isCritical() {
            return critical;
        }
    }
}
