package cn.net.wanmo.common.cert;

import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.crypto.gm.Sm2Util;
import cn.net.wanmo.common.util.Exceptions;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.misc.MiscObjectIdentifiers;
import org.bouncycastle.asn1.misc.NetscapeCertType;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509v3CertificateBuilder;
import org.bouncycastle.jcajce.provider.asymmetric.x509.CertificateFactory;
import org.bouncycastle.jce.X509KeyUsage;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.util.encoders.Base64;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.cert.X509Certificate;
import java.util.Date;

/**
 * 国密证书工具类
 */
public class Sm2CertUtil {

    /**
     * 获取扩展密钥用途
     *
     * @return 增强密钥用法ASN.1对象
     */
    public static DERSequence extendedKeyUsage() {
        // // 客户端身份认证
        // ASN1ObjectIdentifier clientAuth = new ASN1ObjectIdentifier("1.3.6.1.5.5.7.3.2");
        // // 安全电子邮件
        // ASN1ObjectIdentifier emailProtection = new ASN1ObjectIdentifier("1.3.6.1.5.5.7.3.4");
        // 构造容器对象
        ASN1EncodableVector vector = new ASN1EncodableVector();
        // 客户端身份认证
        vector.add(KeyPurposeId.id_kp_clientAuth);
        // 安全电子邮件
        vector.add(KeyPurposeId.id_kp_emailProtection);
        return new DERSequence(vector);
    }



    /**
     * 生成证书文件
     *
     * @param x509Certificate
     *            X.509格式证书
     * @param savePath
     *            证书保存路径
     */
    public static void makeCertFile(X509Certificate x509Certificate, Path savePath) {
        try {
            if (Files.exists(savePath)) {
                Files.deleteIfExists(savePath); // 删除已存在文件
            }
            Files.createFile(savePath); // 创建文件

            System.out.println("savePath: " + savePath.toAbsolutePath().toString());
            // 获取ASN.1编码的证书字节码
            byte[] asn1BinCert = x509Certificate.getEncoded();
            // 编码为BASE64 便于传输
            byte[] base64EncodedCert = Base64.encode(asn1BinCert);
            // 写入文件
            Files.write(savePath, base64EncodedCert);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 生成证书文件，默认有效期 365 天
     * @param privateKey 私钥
     * @param publickey 公钥
     * @param subject 使用者证书主题
     * @param filePath 文件路径
     * @param fileName 文件名称：sm2.cer
     */
    public static void createCertFile(PrivateKey privateKey, PublicKey publickey, X500Name subject, String filePath, String fileName) {
        final X500Name issuer = X500NameUtil.getIssuerSM2();
        Date notBefore = new Date();
        Date notAfter = new Date(System.currentTimeMillis() + 1000 * 60 * 60 * 24 * 365);
        createCertFile(privateKey, publickey, issuer, subject, notBefore, notAfter, filePath, fileName);
    }

    /**
     * 生成证书文件
     * @param privateKey 私钥
     * @param publickey 公钥
     * @param issuer 颁发者证书主题
     * @param subject 使用者证书主题
     * @param notBefore 有效起始
     * @param notAfter 有效截至
     * @param filePath 文件路径
     * @param fileName 文件名称：sm2.cer
     */
    public static void createCertFile(PrivateKey privateKey, PublicKey publickey, X500Name issuer, X500Name subject, Date notBefore, Date notAfter, String filePath, String fileName) {
        try {
            // 证书签名实现类 附加了 SM3WITHSM2 和 PrivateKey
            JcaContentSignerBuilder jcaContentSB = new JcaContentSignerBuilder("SM3withSM2");
            jcaContentSB.setProvider(Sm2Util.provider);
            ContentSigner sigGen = jcaContentSB.build(privateKey);

            // 准备证书信息
            BigInteger sn = BigInteger.valueOf(System.currentTimeMillis());

            // 构造证书信息
            JcaX509v3CertificateBuilder jcaX509v3Cert = new JcaX509v3CertificateBuilder(issuer, sn, notBefore, notAfter, subject, publickey);
            jcaX509v3Cert.addExtension(Extension.keyUsage, false, new X509KeyUsage(X509KeyUsage.digitalSignature | X509KeyUsage.nonRepudiation));
            jcaX509v3Cert.addExtension(Extension.extendedKeyUsage, false, extendedKeyUsage());
            jcaX509v3Cert.addExtension(MiscObjectIdentifiers.netscapeCertType, false, new NetscapeCertType(NetscapeCertType.sslClient));

            // 构造X.509 第3版的证书构建者
            X509v3CertificateBuilder x509v3Cert = jcaX509v3Cert;

            // 将证书构造参数装换为X.509证书对象
            X509Certificate certificate = new JcaX509CertificateConverter().setProvider(Sm2Util.provider).getCertificate(x509v3Cert.build(sigGen));

            // 保存为证书文件
            makeCertFile(certificate, Paths.get(filePath, fileName));
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * SM2 证书验签
     * @param sm2CertFilePath 证书文件路径
     * @param data 原文
     * @param signValue 签名值
     * @return 验签结果
     */
    public static boolean verifyWithCertFile(String sm2CertFilePath, String data, String signValue) {
        File sm2CertFile = new File(sm2CertFilePath);
        return verifyWithCertFile(sm2CertFile, data, signValue);
    }

    /**
     * SM2 证书验签
     * @param sm2CertFile 证书文件
     * @param data 原文
     * @param signValue 签名值
     * @return 验签结果
     */
    public static boolean verifyWithCertFile(File sm2CertFile, String data, String signValue) {
        try {
            // 获取 sm2CertBase64 字符串
            final byte[] bytes = Files.readAllBytes(sm2CertFile.toPath());
//            final String sm2CertBase64 = FileUtil.readFileToString(sm2CertFile, CharsetUtil.DEFAULT);
            return verifyWithCertBase64(new String(bytes), data, signValue);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * SM2 证书验签
     * @param sm2CertBase64 证书 base64
     * @param data 原文
     * @param signValue 签名值
     * @return 验签结果
     */
    public static boolean verifyWithCertBase64(String sm2CertBase64, String data, String signValue) {
        try {
            // 解析证书
            CertificateFactory factory = new CertificateFactory();
            X509Certificate certificate = (X509Certificate) factory.engineGenerateCertificate(new ByteArrayInputStream(Base64.decode(sm2CertBase64)));

            // 验证签名
            Signature signature = Signature.getInstance(certificate.getSigAlgName(), Sm2Util.provider);
            signature.initVerify(certificate);
            signature.update(data.getBytes());

            return signature.verify(CodecUtil.decodeHex(signValue));
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }


}
