package cn.net.wanmo.common.cert;

import cn.net.wanmo.common.util.StringUtil;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.X500NameBuilder;
import org.bouncycastle.asn1.x500.style.BCStyle;

public class X500NameUtil {
    public static final String RSA = "RSA";
    public static final String SM2 = "SM2";

    /**
     * 构造 主题名称
     * @param option 证书信息
     * @return 证书主题
     */
    public static X500Name createX500Name(CertOption option) {
        X500NameBuilder builder = new X500NameBuilder(BCStyle.INSTANCE);
        // 国家代码
        if (StringUtil.isNotBlank(option.getC())) {
            builder.addRDN(BCStyle.C, option.getC());
        }
        // 省份
        if (StringUtil.isNotBlank(option.getS())) {
            builder.addRDN(BCStyle.ST, option.getS());
        }
        // 地区
        if (StringUtil.isNotBlank(option.getL())) {
            builder.addRDN(BCStyle.L, option.getL());
        }
        // 组织
        if (StringUtil.isNotBlank(option.getO())) {
            builder.addRDN(BCStyle.O, option.getO());
        }
        // 身份
        if (StringUtil.isNotBlank(option.getCn())) {
            builder.addRDN(BCStyle.CN, option.getCn());
        }
        // 其他内容
        if (StringUtil.isNotBlank(option.getOu())) {
            builder.addRDN(BCStyle.OU, option.getOu());
        }
        // 描述
        if (StringUtil.isNotBlank(option.getDescription())) {
            builder.addRDN(BCStyle.DESCRIPTION, option.getDescription());
        }

        X500Name subject = builder.build();
        return subject;
    }

    /**
     * 颁发者主题
     * @return 颁发者主题
     */
    public static X500Name getIssuerRSA() {
        return getIssuer(RSA);
    }

    /**
     * 颁发者主题
     * @return 颁发者主题
     */
    public static X500Name getIssuerSM2() {
        return getIssuer(SM2);
    }

    /**
     * 颁发者主题
     * @param description 算法：RSA SM2
     * @return 颁发者主题
     */
    public static X500Name getIssuer(String description) {
        CertOption option = new CertOption();
        option.setS("Shan Xi");
        option.setL("TaiYuan");
        option.setO("WanMo，Inc.");
        option.setO("WanMo Digital Certificate Authority");
        option.setCn("WanMo CA");
        option.setDescription(description);
        return createX500Name(option);
    }

    /**
     * 单位证书主题
     * @param o 单位名称
     * @param ou 统一社会信用代码
     * @return 单位证书主题
     */
    public static X500Name getSubjectForOrganizationRSA(String o, String ou) {
        return getSubjectForOrganization(o, ou, RSA);
    }

    /**
     * 单位证书主题
     * @param o 单位名称
     * @param ou 统一社会信用代码
     * @return 单位证书主题
     */
    public static X500Name getSubjectForOrganizationSM2(String o, String ou) {
        return getSubjectForOrganization(o, ou, SM2);
    }

    /**
     * 单位证书主题
     * @param o 单位名称
     * @param ou 统一社会信用代码
     * @param description 算法：RSA SM2
     * @return 单位证书主题
     */
    public static X500Name getSubjectForOrganization(String o, String ou, String description) {
        return getSubjectForOrganization(o, ou, null, description);
    }


    /**
     * 单位证书主题
     * @param o 单位名称
     * @param ou 统一社会信用代码
     * @param g 单位多个证书标识
     * @return 单位证书主题
     */
    public static X500Name getSubjectForOrganizationRSA(String o, String ou, String g) {
        return getSubjectForOrganization(o, ou, g, RSA);
    }

    /**
     * 单位证书主题
     * @param o 单位名称
     * @param ou 统一社会信用代码
     * @param g 单位多个证书标识
     * @return 单位证书主题
     */
    public static X500Name getSubjectForOrganizationSM2(String o, String ou, String g) {
        return getSubjectForOrganization(o, ou, g, SM2);
    }

    /**
     * 单位证书主题
     * @param o 单位名称
     * @param ou 统一社会信用代码
     * @param g 单位多个证书标识
     * @param description 算法：RSA SM2
     * @return 单位证书主题
     */
    public static X500Name getSubjectForOrganization(String o, String ou, String g, String description) {
        CertOption option = new CertOption();
        option.setS("ShanXi");
        option.setL("TaiYuan");
        option.setO(o);
        option.setOu(ou);
        option.setG(g);
        option.setDescription(description);
        return createX500Name(option);
    }

    /**
     * 个人证书主题
     * @param cn 姓名
     * @param ou 证件号码
     * @return 个人证书主题
     */
    public static X500Name getSubjectForPersonRSA(String cn, String ou) {
        return getSubjectForPerson(cn, ou, RSA);
    }

    /**
     * 个人证书主题
     * @param cn 姓名
     * @param ou 证件号码
     * @return 个人证书主题
     */
    public static X500Name getSubjectForPersonSM2(String cn, String ou) {
        return getSubjectForPerson(cn, ou, SM2);
    }

    /**
     * 个人证书主题
     * @param cn 姓名
     * @param ou 证件号码
     * @param description 算法：RSA SM2
     * @return 个人证书主题
     */
    public static X500Name getSubjectForPerson(String cn, String ou, String description) {
        return getSubjectForPerson(cn, ou, null, description);
    }

    /**
     * 个人证书主题
     * @param cn 姓名
     * @param ou 证件号码
     * @param g 个人多个证书标识
     * @return 个人证书主题
     */
    public static X500Name getSubjectForPersonRSA(String cn, String ou, String g) {
        return getSubjectForPerson(cn, ou, g, RSA);
    }

    /**
     * 个人证书主题
     * @param cn 姓名
     * @param ou 证件号码
     * @param g 个人多个证书标识
     * @return 个人证书主题
     */
    public static X500Name getSubjectForPersonSM2(String cn, String ou, String g) {
        return getSubjectForPerson(cn, ou, g, SM2);
    }

    /**
     * 个人证书主题
     * @param cn 姓名
     * @param ou 证件号码
     * @param g 个人多个证书标识
     * @param description 算法：RSA SM2
     * @return 个人证书主题
     */
    public static X500Name getSubjectForPerson(String cn, String ou, String g, String description) {
        CertOption option = new CertOption();
        option.setS("ShanXi");
        option.setL("TaiYuan");
        option.setCn(cn);
        option.setOu(ou);
        option.setG(g);
        option.setDescription(description);
        return createX500Name(option);
    }
}
