package cn.net.wanmo.common.util;

import com.google.common.io.Resources;
import org.ini4j.Config;
import org.ini4j.Ini;
import org.ini4j.Profile.Section;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URL;

/**
 * INI 格式文件工具类
 */
public class Ini4jUtil {
	private static Logger logger = LoggerFactory.getLogger(Ini4jUtil.class);

	/**
	 * 获取配置文件的 Section <br/>
	 * 默认是 最后一个 section
	 * @param ini 配置
	 * @param sectionName 段名
	 * @return 段
	 */
	public static Section getSection(Ini ini, String sectionName) {
		return getSection(ini, sectionName, null);
	}

	/**
	 * 获取配置文件的 Section
	 *
	 * @param ini 配置
	 * @param sectionName 段名
	 * @param index 索引，第几个 section
	 * @return 段
	 */
	public static Section getSection(Ini ini, String sectionName, Integer index) {
		if (ini == null || StringUtil.isBlank(sectionName)) {
			return null;
		}

		if (index == null) {
			return ini.get(sectionName);
		}

		return ini.get(sectionName, index.intValue());
	}

	/**
	 * 获取默认配置文件的 Section <br>
	 * 获取 class path 路径下的配置文件， 默认 fileName="config/wanmo-config.ini"
	 *
	 * @param sectionName 段名
	 * @param index 索引，第几个 section
	 * @return 段
	 */
	public static Section getSection(String sectionName, Integer index) {
		return getSection(iniDefault, sectionName, index);
	}

	/**
	 * 获取默认配置文件的 Section <br>
	 * 默认是 最后一个 section 的值
	 * 获取 class path 路径下的配置文件， 默认 fileName="config/wanmo-config.ini"
	 *
	 * @param sectionName 段名
	 * @return 段
	 */
	public static Section getSection(String sectionName) {
		return getSection(sectionName, null);
	}

	/**
	 * 获取配置文件的 value
	 *
	 * @param section 段
	 * @param key 键
	 * @return 值
	 */
	public static String getValue(Section section, String key) {
		if (section == null || StringUtil.isBlank(key)) {
			return null;
		}

		return section.get(key);
	}

	/**
	 * 获取 class path 路径下的配置文件
	 *
	 * @return 配置
	 */
	public static Ini getIni(String fileName) {
		Ini ini = null;
		try {
			Config cfg = new Config();
			// 设置Section允许出现重复
			cfg.setMultiSection(true);

			// 生成配置文件的URL
			URL url = Resources.getResource(fileName);

			ini = new Ini();
			ini.setConfig(cfg);
			ini.load(url);
		} catch (Exception e) {
			throw new RuntimeException(String.format("加载配置文件异常： %s", fileName), e);
		}

		return ini;
	}

	/**
	 * 获取 class path 路径下的配置文件， 默认 fileName=wanmo-system-ini.conf
	 */
	public static Ini iniDefault = null;
	static {
		try {
			iniDefault = getIni("config/wanmo-config.ini");
		} catch (Exception e) {
			logger.warn(e.getMessage());
		}
	}
}
