package cn.net.wanmo.common.windows;

import cn.net.wanmo.common.charset.CharsetUtil;
import cn.net.wanmo.common.util.Exceptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;

/**
 * 命令行执行工具类 <br/>
 * cmd /c dir：执行完dir命令后关闭命令窗口； <br/>
 * cmd /k dir：执行完dir命令后不关闭命令窗口。 <br/>
 * cmd /c start dir：会打开一个新窗口后执行dir指令，原窗口会关闭； <br/>
 * cmd /k start dir：会打开一个新窗口后执行dir指令，原窗口不会关闭。 <br/>
 */
public class CmdExecUtil {
    private static Logger logger = LoggerFactory.getLogger(CmdExecUtil.class);

    /**
     * 执行指令
     * @param command 指令
     */
    public static R execCommand(String command) {
        return execCommand(command, false);
    }

    /**
     * 执行指令
     * @param command 指令
     * @param isWait 是否等待
     */
    public static R execCommand(String command, boolean isWait) {
        try {
            final Process process = Runtime.getRuntime().exec(command);
            return getProcessR(process, isWait);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }
    /**
     * 执行指令
     * @param commands 指令数组
     */
    public static R execCommand(String[] commands) {
        return execCommand(commands, false);
    }

    /**
     * 执行指令
     * @param commands 指令数组
     * @param isWait 是否等待
     */
    public static R execCommand(String[] commands, boolean isWait) {
        try {
            final Process process = Runtime.getRuntime().exec(commands);
            return getProcessR(process, isWait);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    public static R getProcessR(Process process, boolean isWait) {
        try {
            R r = new R();
            r.setInputStr(getStreamStr(process.getInputStream()));
            r.setErrorStr(getStreamStr(process.getErrorStream()));
            if (isWait) {
                final int waitFor = process.waitFor();
                r.setWaitFor(waitFor);
            }

            process.destroy();
            return r;
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 接收的数据流取为字符串， charsetName 默认 GBK
     * @param is 数据流
     * @return 字符串
     */
    public static String getStreamStr(InputStream is) {
        return getStreamStr(is, CharsetUtil.GBK_NAME);
    }

    /**
     * 接收的数据流取为字符串
     * @param is 数据流
     * @param charsetName 编码
     * @return 字符串
     */
    public static String getStreamStr(InputStream is, String charsetName) {
        BufferedReader br = null;
        try {
            br = new BufferedReader(new InputStreamReader(is, charsetName));

            StringBuilder sb = new StringBuilder();
            String line;
            while ((line = br.readLine()) != null) {
                sb.append(line);
                sb.append("\n");
            }

            return sb.toString();
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        } finally {
            try {
                is.close();
                br.close();
            } catch (Exception exception) {}
        }
    }

    static class R {
        private int waitFor;
        private String inputStr;
        private String errorStr;

        public R() {
        }

        public R(int waitFor, String inputStr, String errorStr) {
            this.waitFor = waitFor;
            this.inputStr = inputStr;
            this.errorStr = errorStr;
        }

        public int getWaitFor() {
            return waitFor;
        }

        public void setWaitFor(int waitFor) {
            this.waitFor = waitFor;
        }

        public String getInputStr() {
            return inputStr;
        }

        public void setInputStr(String inputStr) {
            this.inputStr = inputStr;
        }

        public String getErrorStr() {
            return errorStr;
        }

        public void setErrorStr(String errorStr) {
            this.errorStr = errorStr;
        }
    }
}
