package cn.net.wanmo.module.sys.util;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.List;

import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.util.*;
import cn.net.wanmo.plugin.spring.util.SpringUtil;
import cn.net.wanmo.util.PropUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.web.multipart.MultipartFile;

import cn.net.wanmo.module.sys.entity.Annex;
import cn.net.wanmo.module.sys.service.AnnexService;

import javax.servlet.http.HttpServletRequest;

/**
 * 附件工具类
 */
public class AnnexUtil {
    private static final Logger logger = LoggerFactory.getLogger(AnnexUtil.class);
    private static AnnexService annexService = SpringUtil.getBean(AnnexService.class);


    /**
     * MultipartFile 对象转为 Annex
     *
     * @param file
     * @return
     */
    public static Annex toAnnex(MultipartFile file) {
        Annex annex = new Annex();

        annex.setIsNewRecord(true);
        annex.setId(IdGen.uuid());

        annex.setOriginalFilename(file.getOriginalFilename());
        annex.setContentType(file.getContentType());
        annex.setFileSize(file.getSize());
        annex.setFileName(FileUtil.getFileNameNoSuffix(annex.getOriginalFilename()));
        annex.setFileType("0"); // 默认附件类型都是 0
        annex.setDeleteStatus(Annex.STATUS_REMOVE); // 默认是移除状态，只有业务数据真正保存的时候才会改为正常状态

        { // 设置附件保存的相对物理磁盘路径
            String userId = UserUtil.getPrincipal() == null ? "noid" : UserUtil.getPrincipal().getId();
            String relativePath = userId + "/" + DateUtil.formatDay() + "/"; // 相对路径

            annex.setFilePath(relativePath + annex.getPhysicalName());
        }

        return annex;
    }

    /**
     * 保存上传的文件到物理磁盘
     *
     * @param file
     * @param annex
     * @throws IOException
     */
    public static void transferFile(MultipartFile file, Annex annex) throws IOException {
        File targetFile = new File(getUploadBaseDir() + annex.getFilePath()); // 目标文件夹
        if (targetFile.getParentFile().exists()) {
            // 存在则不处理
        } else { // 文件夹不存在，则创建
            targetFile.getParentFile().mkdirs();
        }
        file.transferTo(targetFile); // 保存文件
    }

    /**
     * 清理垃圾附件
     */
    public static void clearRubbishAnnex() {
        Annex entity = new Annex();
        entity.setDeleteStatus(Annex.STATUS_REMOVE);
        List<Annex> list = annexService.findList(entity);

        for (Annex annex : list) {
            Date updateTime = annex.getUpdateTime();
            // 清理时间超过 30 分钟的附件
            if (DateUtil.nowLong() - updateTime.getTime() > 1000 * 60 * 30) {
                try {
                    FileUtil.forceDeleteOnExit(new File(annex.getFilePath()));
                    annexService.delete(annex.getId());
                } catch (IOException e) {
                }
            }
        }
    }


    /**
     * 获取文件上传的基础路径
     */
    public static String getUploadBaseDir() {
        String basePath = PropUtil.getAnnexUploadBasedir();

        if (StringUtil.isBlank(basePath)) {
            basePath = SpringUtil.getRequest().getServletContext().getRealPath("/");
        }

        if (basePath.endsWith("/")) {
        } else {
            basePath += "/";
        }

        basePath = basePath + "upload/";

        return basePath;
    }

    /**
     * 获取文件下载时需要的文件名<br>
     * 处理以后的文件名，解决获得中文参数的乱码
     *
     * @param request
     * @param filename
     * @return
     * @throws UnsupportedEncodingException
     */
    public static String getDownFileName(HttpServletRequest request, String filename) throws UnsupportedEncodingException {
        // 获得请求头中的User-Agent
        String agent = request.getHeader(HttpHeaders.USER_AGENT);

        // 根据不同的客户端进行不同的编码
        String filenameEncoder = "";
        if (agent.contains("MSIE")) {
            // IE浏览器
            filenameEncoder = URLEncoder.encode(filename, "utf-8");
            filenameEncoder = filenameEncoder.replace("+", " ");
        } else if (agent.contains("Firefox")) {
            // 火狐浏览器
            filenameEncoder = "=?utf-8?B?" + CodecUtil.encodeBase64(filename.getBytes("utf-8")) + "?=";
        } else {
            // 其它浏览器
            filenameEncoder = URLEncoder.encode(filename, "utf-8");
        }
        return filenameEncoder;
    }
}
