package cn.net.wanmo.module.sys.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import cn.net.wanmo.common.util.BooleanUtil;
import cn.net.wanmo.plugin.spring.util.SpringUtil;
import cn.net.wanmo.module.sys.util.RoleUtil;
import com.google.common.collect.Lists;

import cn.net.wanmo.common.util.CollectionUtil;
import cn.net.wanmo.common.util.StringUtil;
import cn.net.wanmo.module.sys.entity.Menu;
import cn.net.wanmo.module.sys.service.MenuService;
import cn.net.wanmo.module.sys.entity.Role;
import cn.net.wanmo.module.sys.entity.User;
import org.apache.shiro.subject.Subject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 菜单工具类
 */
public class MenuUtil {
	private static final Logger logger = LoggerFactory.getLogger(MenuUtil.class);
	private static MenuService menuService = SpringUtil.getBean(MenuService.class);

	/**
	 * 查找所有菜单信息
	 *
	 * @return
	 */
	public static List<Menu> findListAll() {
		return menuService.findList(new Menu().getDefaultOrderBy());
	}

	/**
	 * 获取菜单子节点
	 *
	 * @param parentId
	 * @return
	 */
	public static List<Menu> findListByParentId(String parentId) {
		List<Menu> list = new ArrayList<Menu>();

		for (Menu menu : findListAll()) {
			if (StringUtil.equals(parentId, menu.getParentId())) {
				list.add(menu);
			}
		}

		return list;
	}

	/**
	 * 返回用户拥有的的所有菜单，去重以后的<br>
	 * 超级用户，则拥有所有权限<br>
	 * 未启用的角色的菜单是不会返回的
	 */
	public static List<Menu> findListByUser(String userId) {
		return findListByUser(new User(userId));
	}

	/**
	 * 返回用户拥有的的所有菜单，去重以后的<br>
	 * 超级用户，则拥有所有权限<br>
	 * 未启用的角色的菜单是不会返回的
	 */
	public static List<Menu> findListByUser(User user) {
		List<Menu> returnList = Lists.newArrayList();

		if (user == null || StringUtil.isBlank(user.getId())) {
			return returnList;
		}

		if (StringUtil.equals(User.SUPER_USER_ID, user.getId())) { // 如果是超级用户，则拥有所有权限
			returnList = findListAll();
		} else {
			List<Role> listRole = RoleUtil.findListByUser(user); // 用户拥有的角色

			if (CollectionUtil.isNotEmpty(listRole)) {

				for (Role r : listRole) { // 遍历角色，添加菜单信息
					List<Menu> listMenu = menuService.findListByRoleId(r.getId()); // 当前角色拥有的菜单信息

					for (Menu m : listMenu) {
						if (isContainMenu(returnList, m)) { // 如果包含则不添加继续遍历
							continue;
						}
						returnList.add(m);
					}
				}
			}
		}

		Collections.sort(returnList); // 重新排序
		return returnList;
	}

	/**
	 * 菜单列表中是否包含该菜单
	 *
	 * @param menuList 菜单列表
	 * @param menu     判断是否包含的菜单
	 * @return 包含 true, 不包含 false
	 */
	public static Boolean isContainMenu(List<Menu> menuList, Menu menu) {
		boolean flag = false;

		if (CollectionUtil.isEmpty(menuList) || menu == null) {
			flag = false;
		} else {
			for (Menu e : menuList) {
				if (StringUtil.equals(e.getId(), menu.getId())) {
					flag = true;
					break;
				}

			}
		}

		return flag;
	}

	/**
	 * 移除所有菜单缓存信息
	 *
	 * @return
	 */
	public static void clearCache() {
	}

	/**
	 * 只返回当前用户拥有权限的菜单
	 * @param menuList 所有的菜单
	 * @return 用户拥有权限，并允许显示的菜单
	 */
	public static List<Menu> hasPermissionMenus(List<Menu> menuList) {
		List<Menu> returnList = new ArrayList<>();

		final Subject subject = UserUtil.getSubject();
		for (Menu menu : menuList) {
			if (StringUtil.isBlank(menu.getPermission())) {
				continue;
			}

			final boolean isShow = BooleanUtil.toBoolean(menu.getIsShow().getValue());
			final boolean permitted = subject.isPermitted(menu.getPermission());

			if (isShow && permitted) {
				returnList.add(menu);
			}
		}

		return returnList;
	}

}
