package cn.net.wanmo.security.filter;

import cn.net.wanmo.common.crypto.rsa.RsaUtil;
import cn.net.wanmo.common.util.Exceptions;
import cn.net.wanmo.common.util.StringUtil;
import cn.net.wanmo.security.token.UsernamePasswordToken;
import cn.net.wanmo.util.PropUtil;
import org.apache.shiro.authc.*;
import org.apache.shiro.subject.Subject;
import org.apache.shiro.web.util.WebUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;

/**
 * 表单登录过滤器
 */
public class FormAuthenticationFilter extends org.apache.shiro.web.filter.authc.FormAuthenticationFilter {
	Logger logger = LoggerFactory.getLogger(getClass());

	public static final String DEFAULT_CAPTCHA_PARAM = "validateCode";
	public static final String DEFAULT_MESSAGE_PARAM = "message";

	/**
	 * 可以根据不同的登录方式返回不同的 token
	 */
	@Override
	protected AuthenticationToken createToken(ServletRequest request, ServletResponse response) {
		String username = getUsername(request);
		String password = getPassword(request);
		password = RsaUtil.decryptByPri(password, PropUtil.priKeyBase64);

		String host = getHost(request);
		boolean rememberMe = isRememberMe(request);

		return new UsernamePasswordToken(username, password, rememberMe, host, null);
	}

	/**
	 * 获取登录用户名，如果获取不到则返回空
	 */
	@Override
	protected String getUsername(ServletRequest request) {
		String username = super.getUsername(request);

		if (StringUtil.isBlank(username)) {
			username = StringUtil.toString(request.getAttribute(getUsernameParam()), StringUtil.EMPTY);
		}
		return username;
	}

	/**
	 * 获取登录密码，如果获取不到则返回空
	 */
	@Override
	protected String getPassword(ServletRequest request) {
		String password = super.getPassword(request);

		if (StringUtil.isBlank(password)) {
			password = StringUtil.toString(request.getAttribute(getPasswordParam()), StringUtil.EMPTY);
		}
		return password;
	}

	/**
	 * 获取记住我
	 */
	@Override
	protected boolean isRememberMe(ServletRequest request) {
		String isRememberMe = WebUtils.getCleanParam(request, getRememberMeParam());

		if (StringUtil.isBlank(isRememberMe)) {
			isRememberMe = StringUtil.toString(request.getAttribute(getRememberMeParam()), StringUtil.EMPTY);
		}
		return Boolean.parseBoolean(isRememberMe);
	}

	/**
	 * 验证码
	 *
	 * @param request
	 * @return
	 */
	protected String getCaptcha(ServletRequest request) {
		return WebUtils.getCleanParam(request, getCaptchaParam());
	}

	@Override
	public String getUsernameParam() {
		return "loginName";
	}

	@Override
	public String getPasswordParam() {
		return "loginPswd";
	}

	public String getCaptchaParam() {
		return DEFAULT_CAPTCHA_PARAM;
	}

	public String getMessageParam() {
		return DEFAULT_MESSAGE_PARAM;
	}

	/**
	 * 登录成功之后跳转URL
	 */
	@Override
	public String getSuccessUrl() {
		return super.getSuccessUrl();
	}

	/**
	 * 当登录成功
	 */
	@Override
	protected boolean onLoginSuccess(AuthenticationToken token, Subject subject, ServletRequest request,
			ServletResponse response) throws Exception {
		// 获取session中的请求信息，并且清理他， 这样就跳转到指定的登录成功页面了
		WebUtils.getAndClearSavedRequest(request);
		return super.onLoginSuccess(token, subject, request, response);
	}

	/**
	 * 当登录失败
	 */
	@Override
	protected boolean onLoginFailure(AuthenticationToken token, AuthenticationException e, ServletRequest request,
			ServletResponse response) {
		String message = "";
		if (e instanceof IncorrectCredentialsException) {
			message = "用户或密码错误, 请重试.";
		} else if (e instanceof UnknownAccountException) {
			message = "账号不存在.";
		} else if (e instanceof DisabledAccountException) {
			message = "禁用的账号";
		} else if (e instanceof ExpiredCredentialsException) {
			message = "凭证过期";
		} else if (e instanceof ExcessiveAttemptsException) {
			message = "认证次数超过限制";
		} else if (e instanceof LockedAccountException) {
			message = "账号被锁定";
		} else {
			message = "系统出现点问题，请稍后再试！";
			logger.error(Exceptions.getStackTraceAsString(e)); // 输出到错误日志
		}
		request.setAttribute(getMessageParam(), message);
		return true;
	}

	/**
	 * 表示当访问拒绝时是否已经处理了；如果返回true表示需要继续处理；如果返回false表示该拦截器实例已经处理了，将直接返回即可。
	 * onAccessDenied是否执行取决于isAccessAllowed的值，如果返回true则onAccessDenied不会执行；如果返回false，执行onAccessDenied
	 * 如果onAccessDenied也返回false，则直接返回，不会进入请求的方法（只有isAccessAllowed和onAccessDenied的情况下）
	 */
	@Override
	protected boolean onAccessDenied(ServletRequest request, ServletResponse response) throws Exception {
		return super.onAccessDenied(request, response);
	}

}
