package cn.net.wanmo.security.realm;

import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.security.PasswordUtil;
import cn.net.wanmo.common.util.*;
import cn.net.wanmo.module.sys.entity.Menu;
import cn.net.wanmo.module.sys.entity.Role;
import cn.net.wanmo.module.sys.entity.User;
import cn.net.wanmo.module.sys.service.UserService;
import cn.net.wanmo.module.sys.util.MenuUtil;
import cn.net.wanmo.module.sys.util.RoleUtil;
import cn.net.wanmo.security.Principal;
import org.apache.shiro.authc.*;
import org.apache.shiro.authc.credential.HashedCredentialsMatcher;
import org.apache.shiro.authz.AuthorizationInfo;
import org.apache.shiro.authz.Permission;
import org.apache.shiro.authz.SimpleAuthorizationInfo;
import org.apache.shiro.realm.AuthorizingRealm;
import org.apache.shiro.subject.PrincipalCollection;
import org.apache.shiro.util.ByteSource;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.util.Collection;
import java.util.List;

/**
 * 登录名认证
 */
public class LoginNameRealm extends AuthorizingRealm {

	@Autowired
	private UserService userService;

	/**
	 * 获取授权信息
	 */
	@Override
	protected AuthorizationInfo doGetAuthorizationInfo(PrincipalCollection principals) {
		SimpleAuthorizationInfo info = new SimpleAuthorizationInfo();

		Principal principal = (Principal) getAvailablePrincipal(principals);

		if (CollectionUtil.isNotEmpty(principal.getRoleList())) {
			for (Role r : principal.getRoleList()) {
				if (StringUtil.isNotBlank(r.getEname())) {
					info.addRole(r.getEname());
				}
			}
		}

		if (CollectionUtil.isNotEmpty(principal.getMenuList())) {
			for (Menu m : principal.getMenuList()) {
				if (StringUtil.isNotBlank(m.getPermission())) {
					info.addStringPermission(m.getPermission());
				}
			}
		}

		return info;
	}

	/**
	 * 获取认证信息
	 */
	@Override
	protected AuthenticationInfo doGetAuthenticationInfo(AuthenticationToken token) throws AuthenticationException {
		UsernamePasswordToken authcToken = (UsernamePasswordToken) token;
		String loginName = authcToken.getUsername();

		User user = userService.getByLoginName(loginName);

		if (user == null) {
			throw new UnknownAccountException("用户 " + loginName + " 不存在");
		}
		if (BooleanUtil.toBoolean(user.getEnable().getValue())) {
		} else {
            throw new DisabledAccountException("该账号已禁止使用");
		}

		Principal principal = new Principal(user);
		principal.setRoleList(RoleUtil.findListByUser(user)); // 设置用户拥有的角色
		principal.setMenuList(MenuUtil.findListByUser(user)); // 设置用户拥有的菜单

		byte[] salt = CodecUtil.decodeHex(user.getLoginPswd().substring(0, 16));
		return new SimpleAuthenticationInfo(principal, user.getLoginPswd().substring(16), ByteSource.Util.bytes(salt),
				getName());
	}

	@Override
	protected void checkPermission(Permission permission, AuthorizationInfo info) {
		authorizationValidate(permission);
		super.checkPermission(permission, info);
	}

	@Override
	protected boolean[] isPermitted(List<Permission> permissions, AuthorizationInfo info) {
		if (permissions != null && !permissions.isEmpty()) {
			for (Permission permission : permissions) {
				authorizationValidate(permission);
			}
		}
		return super.isPermitted(permissions, info);
	}

	@Override
	public boolean isPermitted(PrincipalCollection principals, Permission permission) {
		authorizationValidate(permission);
		return super.isPermitted(principals, permission);
	}

	@Override
	protected boolean isPermittedAll(Collection<Permission> permissions, AuthorizationInfo info) {
		if (permissions != null && !permissions.isEmpty()) {
			for (Permission permission : permissions) {
				authorizationValidate(permission);
			}
		}
		return super.isPermittedAll(permissions, info);
	}

	/**
	 * 授权验证方法
	 *
	 * @param permission
	 */
	private void authorizationValidate(Permission permission) {
		// TODO 模块授权预留接口
	}

	/**
	 * 设定密码校验的Hash算法与迭代次数
	 */
	@PostConstruct
	public void initCredentialsMatcher() {
		HashedCredentialsMatcher matcher = new HashedCredentialsMatcher(PasswordUtil.HASH_ALGORITHM);
		matcher.setHashIterations(PasswordUtil.HASH_INTERATIONS);
		setCredentialsMatcher(matcher);
	}

	@Override
	protected void clearCache(PrincipalCollection principals) {
		super.clearCache(principals);
	}
}
