package cn.net.wanmo.security.session;

import cn.net.wanmo.common.util.DateUtil;
import cn.net.wanmo.common.util.ObjectUtil;
import cn.net.wanmo.common.util.StringUtil;
import cn.net.wanmo.util.PropUtil;
import cn.net.wanmo.util.Servlets;
import com.google.common.collect.Sets;
import org.apache.shiro.session.Session;
import org.apache.shiro.session.UnknownSessionException;
import org.apache.shiro.session.mgt.eis.EnterpriseCacheSessionDAO;
import org.apache.shiro.subject.PrincipalCollection;
import org.apache.shiro.subject.support.DefaultSubjectContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import java.io.Serializable;
import java.util.Collection;
import java.util.Set;

public class CacheSessionDAO extends EnterpriseCacheSessionDAO implements SessionDAO {
	private final Logger logger = LoggerFactory.getLogger(getClass());

	public CacheSessionDAO() {
		super();
	}

	@Override
	protected Serializable doCreate(Session session) {
		HttpServletRequest request = Servlets.getRequest();
		if (ObjectUtil.isNotNull(request)) {
			String uri = request.getServletPath();
			if (Servlets.isStaticFile(uri)) { // 如果是静态文件，则不创建SESSION
				return null;
			}
		}

		super.doCreate(session);
		logger.debug("createSession {} {}", session, ObjectUtil.isNull(request) ? "" : request.getRequestURI());
		return session.getId();
	}

	@Override
	protected void doUpdate(Session session) {
		if (session == null || session.getId() == null) {
			return;
		}

		HttpServletRequest request = Servlets.getRequest();
		if (ObjectUtil.isNotNull(request)) {
			String uri = request.getServletPath();
			// 如果是静态文件，则不更新SESSION
			if (Servlets.isStaticFile(uri)) {
				return;
			}
			// 如果是视图文件，则不更新SESSION
			if (StringUtil.startsWith(uri, PropUtil.getValue("web.view.prefix")) && StringUtil.endsWith(uri, PropUtil.getValue("web.view.suffix"))) {
				return;
			}
		}

		super.doUpdate(session);
//		logger.debug("updateSession {} {}", session.getId(), ObjectUtil.isNull(request) ? "" : request.getRequestURI());
	}

	@Override
	protected void doDelete(Session session) {
		if (session == null || session.getId() == null) {
			return;
		}

		super.doDelete(session);
		logger.debug("deleteSession {} ", session.getId());
	}

	@Override
	protected Session doReadSession(Serializable sessionId) {
		return super.doReadSession(sessionId);
	}

	@Override
	public Session readSession(Serializable sessionId) throws UnknownSessionException {
		try {
			Session s = null;
			HttpServletRequest request = Servlets.getRequest();
			if (ObjectUtil.isNotNull(request)) {
				String uri = request.getServletPath();
				// 如果是静态文件，则不获取SESSION
				if (Servlets.isStaticFile(uri)) {
					return null;
				}
				s = ObjectUtil.cast(request.getAttribute("session_" + sessionId));
			}
			if (ObjectUtil.isNotNull(s)) {
				return s;
			}

			Session session = super.readSession(sessionId);
//			logger.debug("readSession {} {}", sessionId, ObjectUtil.isNull(request) ? "" : request.getRequestURI());

			if (ObjectUtil.isNotNull(request) && ObjectUtil.isNotNull(session)) {
				request.setAttribute("session_" + sessionId, session);
			}

			return session;
		} catch (UnknownSessionException e) {
			return null;
		}
	}

	/**
	 * 获取活动会话
	 *
	 * @param includeLeave 是否包括离线（最后访问时间大于3分钟为离线会话）
	 * @return Session 集合
	 */
	@Override
	public Collection<Session> getActiveSessions(boolean includeLeave) {
		return getActiveSessions(includeLeave, null, null);
	}

	/**
	 * 获取活动会话
	 *
	 * @param includeLeave  是否包括离线（最后访问时间大于3分钟为离线会话）
	 * @param principal     根据登录者对象获取活动会话
	 * @param filterSession 不为空，则过滤掉（不包含）这个会话。
	 * @return Session集合
	 */
	@Override
	public Collection<Session> getActiveSessions(boolean includeLeave, Object principal, Session filterSession) {
		// 如果包括离线，并无登录者条件。
		if (includeLeave && principal == null) {
			return getActiveSessions();
		}

		Set<Session> sessions = Sets.newHashSet();
		for (Session session : getActiveSessions()) {
			boolean isActiveSession = false;
			// 不包括离线并符合最后访问时间小于等于3分钟条件。
			if (includeLeave || DateUtil.pastMinutes(session.getLastAccessTime()) <= 3) {
				isActiveSession = true;
			}

			// 符合登陆者条件。
			if (ObjectUtil.isNotNull(principal)) {
				PrincipalCollection pc = ObjectUtil.cast(session.getAttribute(DefaultSubjectContext.PRINCIPALS_SESSION_KEY));
				if (principal.toString().equals(ObjectUtil.isNull(pc) ? StringUtil.EMPTY : pc.getPrimaryPrincipal().toString())) {
					isActiveSession = true;
				}
			}

			// 过滤掉的SESSION
			if (ObjectUtil.isNotNull(filterSession) && filterSession.getId().equals(session.getId())) {
				isActiveSession = false;
			}

			if (isActiveSession) {
				sessions.add(session);
			}
		}
		return sessions;
	}

}
