package cn.net.wanmo.common.crypto;

import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.util.Exceptions;
import cn.net.wanmo.common.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.security.SecureRandom;

/**
 * AES对称加密，对明文进行加密、解密处理
 */
public class AesUtil {
    static Logger logger = LoggerFactory.getLogger(AesUtil.class);

    /**
     * ECB模式-加密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要进行加密的原文
     * @param keyStr 进行了Base64编码的秘钥
     * @return String 数据密文，加密后的数据，进行了Base64的编码
     */
    public static String encrypt(String data, String keyStr) {
        return encrypt(data, keyStr, false);
    }

    /**
     * ECB模式-加密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要进行加密的原文
     * @param keyStr 进行了Base64编码的秘钥
     * @param isDecodeBase64   是否对 key 和 iv 进行 Base64 解码
     * @return String 数据密文，加密后的数据，进行了Base64的编码
     */
    public static String encrypt(String data, String keyStr, boolean isDecodeBase64) {
        byte[] bytesData = StringUtil.getBytes(data);
        byte[] bytesKey = null;

        if (isDecodeBase64) {
            bytesKey = CodecUtil.decodeBase64(keyStr);
        } else  {
            bytesKey = StringUtil.getBytes(keyStr);
        }

        return encrypt(bytesData, bytesKey);
    }

    /**
     * ECB模式-加密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要进行加密的原文
     * @param keyStr 进行了Base64编码的秘钥
     * @return String 数据密文，加密后的数据，进行了Base64的编码
     */
    public static String encrypt(byte[] data, byte[] keyStr) {
        return encrypt(data, keyStr, null);
    }

    /**
     * CBC模式-加密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要进行加密的原文
     * @param keyStr 进行了Base64编码的秘钥
     * @param ivStr  进行了Base64编码的偏移量
     * @return String 数据密文，加密后的数据，进行了Base64的编码
     */
    public static String encrypt(String data, String keyStr, String ivStr) {
        return encrypt(data, keyStr, ivStr, false);
    }

    /**
     * CBC模式-加密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要进行加密的原文
     * @param keyStr 进行了Base64编码的秘钥
     * @param ivStr  进行了Base64编码的偏移量
     * @param isDecodeBase64   是否对 key 和 iv 进行 Base64 解码
     * @return String 数据密文，加密后的数据，进行了Base64的编码
     */
    public static String encrypt(String data, String keyStr, String ivStr, boolean isDecodeBase64) {
        byte[] bytesData = StringUtil.getBytes(data);
        byte[] bytesKey = null;
        byte[] bytesIv = null;

        if (isDecodeBase64) {
            bytesKey = CodecUtil.decodeBase64(keyStr);
            bytesIv = CodecUtil.decodeBase64(ivStr);
        } else {
            bytesKey = StringUtil.getBytes(keyStr);
            bytesIv = StringUtil.getBytes(ivStr);
        }

        return encrypt(bytesData, bytesKey, bytesIv);
    }

    /**
     * CBC模式-加密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要进行加密的原文
     * @param keyStr 进行了Base64编码的秘钥
     * @param ivStr  进行了Base64编码的偏移量
     * @return String 数据密文，加密后的数据，进行了Base64的编码
     */
    public static String encrypt(byte[] data, byte[] keyStr, byte[] ivStr) {
        try {
            Cipher cipher = getCipher(Cipher.ENCRYPT_MODE, keyStr, ivStr);

            byte[] encryptBytes = cipher.doFinal(data);
            return CodecUtil.encodeBase64(encryptBytes);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * ECB模式-解密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要解密的数据（数据必须是通过加密后，对加密数据Base64编码的数据）
     * @param keyStr 进行了Base64编码的秘钥
     * @return String 返回解密后的原文
     */
    public static String decrypt(String data, String keyStr) {
        return decrypt(data, keyStr, false);
    }

    /**
     * ECB模式-解密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要解密的数据（数据必须是通过加密后，对加密数据Base64编码的数据）
     * @param keyStr 进行了Base64编码的秘钥
     * @param isDecodeBase64   是否对 key 和 iv 进行 Base64 解码
     * @return String 返回解密后的原文
     */
    public static String decrypt(String data, String keyStr, boolean isDecodeBase64) {
        byte[] dataBytes = CodecUtil.decodeBase64(data);
        byte[] keyBytes = null;
        if (isDecodeBase64) {
            keyBytes = CodecUtil.decodeBase64(keyStr);
        } else {
            keyBytes = StringUtil.getBytes(keyStr);
        }
        return decrypt(dataBytes, keyBytes);
    }

    /**
     * ECB模式-解密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要解密的数据（数据必须是通过加密后，对加密数据Base64编码的数据）
     * @param keyStr 进行了Base64编码的秘钥
     * @return String 返回解密后的原文
     */
    public static String decrypt(byte[] data, byte[] keyStr) {
        return decrypt(data, keyStr, null);
    }

    /**
     * CBC模式-解密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要解密的数据（数据必须是通过加密后，对加密数据Base64编码的数据）
     * @param keyStr 进行了Base64编码的秘钥
     * @param ivStr  进行了Base64编码的偏移量
     * @return String 返回解密后的原文
     */
    public static String decrypt(String data, String keyStr, String ivStr) {
        return decrypt(data, keyStr, ivStr, false);
    }

    /**
     * CBC模式-解密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要解密的数据（数据必须是通过加密后，对加密数据Base64编码的数据）
     * @param keyStr 进行了Base64编码的秘钥
     * @param ivStr  进行了Base64编码的偏移量
     * @param isDecodeBase64   是否对 key 和 iv 进行 Base64 解码
     * @return String 返回解密后的原文
     */
    public static String decrypt(String data, String keyStr, String ivStr, boolean isDecodeBase64) {
        byte[] dataBytes = CodecUtil.decodeBase64(data);
        byte[] keyBytes = null;
        byte[] ivBytes = null;

        if (isDecodeBase64) {
            keyBytes = CodecUtil.decodeBase64(keyStr);
            ivBytes = CodecUtil.decodeBase64(ivStr);
        } else {
            keyBytes = StringUtil.getBytes(keyStr);
            ivBytes = StringUtil.getBytes(ivStr);
        }

        return decrypt(dataBytes, keyBytes, ivBytes);
    }

    /**
     * CBC模式-解密操作： CBC 模式可以用IV， ECB 模式不能用 IV
     *
     * @param data   需要解密的数据（数据必须是通过加密后，对加密数据Base64编码的数据）
     * @param keyStr 进行了Base64编码的秘钥
     * @param ivStr  进行了Base64编码的偏移量
     * @return String 返回解密后的原文
     */
    public static String decrypt(byte[] data, byte[] keyStr, byte[] ivStr) {
        try {
            Cipher cipher = getCipher(Cipher.DECRYPT_MODE, keyStr, ivStr);

            byte[] decryptBytes = cipher.doFinal(data);
            return StringUtil.toString(decryptBytes);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }


    /**
     * 密钥算法 AES
     */
    private static final String KEY_ALGORITHM_AES = "AES";
    /**
     * 密码算法 ECB 128  (AES/ECB/PKCS5Padding) (AES/ECB/NoPadding)
     */
    private static final String CIPHER_ALGORITHM_ECB = "AES/ECB/PKCS5Padding"; // "算法/模式/补码方式"
    /**
     * 密码算法 CBC 128  (AES/CBC/PKCS5Padding) (AES/CBC/NoPadding)
     */
    private static final String CIPHER_ALGORITHM_CBC = "AES/CBC/PKCS5Padding"; // "算法/模式/补码方式"

    /**
     * 获取加密对象
     *
     * @param mode 模式：加密模式 / 解密模式
     * @param key  密钥
     * @param iv   偏移量
     * @param isDecodeBase64   是否对 key 和 iv 进行 Base64 解码
     * @return 加密对象
     * @throws Exception 异常
     */
    private static Cipher getCipher(int mode, String key, String iv, boolean isDecodeBase64) {
        byte[] keyBytes = null;
        byte[] ivBytes = null;

        if (isDecodeBase64) {
            keyBytes = CodecUtil.decodeBase64(key);
            ivBytes = CodecUtil.decodeBase64(iv);
        } else {
            keyBytes = StringUtil.getBytes(key);
            ivBytes = StringUtil.getBytes(iv);
        }

        return getCipher(mode, keyBytes, ivBytes);
    }

    /**
     * 获取加密对象
     *
     * @param mode 模式：加密模式 / 解密模式
     * @param key  密钥
     * @param iv   偏移量
     * @return 加密对象
     * @throws Exception 异常
     */
    private static Cipher getCipher(int mode, byte[] key, byte[] iv) {
        try {
            // 创建加密规则
            final SecretKeySpec secretKeySpec = new SecretKeySpec(key, KEY_ALGORITHM_AES);

            // 创建加密对象
            Cipher cipher = null;
            if (iv == null) {
                cipher = Cipher.getInstance(CIPHER_ALGORITHM_ECB); // "算法/模式/补码方式"
                cipher.init(mode, secretKeySpec); // 初始化加密模式和算法
            } else {
                cipher = Cipher.getInstance(CIPHER_ALGORITHM_CBC); // "算法/模式/补码方式"
                IvParameterSpec ivParameterSpec = new IvParameterSpec(iv); // 创建 iv 向量， 是使用到 CBC 加密模式
                cipher.init(mode, secretKeySpec, ivParameterSpec); // 初始化加密模式和算法
            }

            return cipher;
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }


    /**
     * 生成AES的秘钥，秘钥进行了Base64编码的字符串
     *
     * @return String 对生成的秘钥进行了Base64编码的字符串
     */
    public static String keyGenerate() {
        try {
            KeyGenerator keyGenerator = KeyGenerator.getInstance(KEY_ALGORITHM_AES);
            keyGenerator.init(new SecureRandom());
            SecretKey secretKey = keyGenerator.generateKey();
            byte[] keyBytes = secretKey.getEncoded();

            return CodecUtil.encodeBase64(keyBytes);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 生成随机的密钥 key
     *
     * @return 随机密钥 key
     * @throws Exception 异常
     */
    public static String generateKey() {
        return keyGenerate();
    }

    /**
     * 生成随机的偏移量 iv
     *
     * @return 随机的偏移量 iv
     * @throws Exception 异常
     */
    public static String generateIv() {
        return keyGenerate();
    }

    /**
     * 生成随机的密钥 key
     *
     * @return 随机密钥 key
     * @throws Exception 异常
     */
    public static String getRandomKey() {
        final String randomCode = StringUtil.getRandomCode(16);
        return CodecUtil.encodeBase64(randomCode);
    }

    /**
     * 生成随机的偏移量 iv
     *
     * @return 随机的偏移量 iv
     * @throws Exception 异常
     */
    public static String getRandomIv() {
        final String randomCode = StringUtil.getRandomCode(16);
        return CodecUtil.encodeBase64(randomCode);
    }

}
