package cn.net.wanmo.common.crypto;

import cn.net.wanmo.common.charset.CharsetUtil;
import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.util.Exceptions;
import cn.net.wanmo.common.util.StringUtil;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.security.SecureRandom;

/**
 * 对称加解密
 */
public class DesUtil {

    /**
     * 数据加密, ECB 模式
     *
     * @param data 原文
     * @param key   密钥，必须是8个字节，如 "12345678"
     * @return Base64 的密文
     * @throws Exception 异常
     */
    public static String encipher(String data, String key) {
        return encipher(data, key, null);
    }

    /**
     * 数据加密, CBC 模式
     *
     * @param data 原文
     * @param key   密钥，必须是8个字节，如 "12345678"
     * @param iv    偏移量，必须是8个字节，如 "12345678"
     * @return Base64 的密文
     * @throws Exception 异常
     */
    public static String encipher(String data, String key, String iv) {
        try {
            // 获取加密对象
            final Cipher cipher = getCipher(Cipher.ENCRYPT_MODE, key, iv);
            // 调用加密方法，使用默认字符集获取原文的字节数组
            final byte[] bytes = cipher.doFinal(data.getBytes(CharsetUtil.DEFAULT));
            // 如果直接打印密文会出现乱码，是因为在编码表上面找不到对于的字符,一般需要结合base64
            return CodecUtil.encodeBase64(bytes);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }


    /**
     * 数据解密, ECB 模式
     *
     * @param data Base64 的密文
     * @param key   密钥，必须是8个字节，如 "12345678"
     * @return 原文
     * @throws Exception 异常
     */
    public static String decrypt(String data, String key) {
        return decrypt(data, key, null);
    }

    /**
     * 数据解密, CBC 模式
     *
     * @param data Base64 的密文
     * @param key   密钥，必须是8个字节，如 "12345678"
     * @param iv    偏移量，必须是8个字节，如 "12345678"
     * @return 原文
     * @throws Exception 异常
     */
    public static String decrypt(String data, String key, String iv) {
        try {
            // 获取加密对象
            final Cipher cipher = getCipher(Cipher.DECRYPT_MODE, key, iv);
            // 调用解密方法，参数表示密文的字节数组
            final byte[] bytes = cipher.doFinal(CodecUtil.decodeBase64(data));
            // 使用默认字符集，返回字符串
            return StringUtil.toEncodedString(bytes, CharsetUtil.DEFAULT);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }



    /**
     * 密钥算法 AES
     */
    private static final String KEY_ALGORITHM_DES = "DES";
    /**
     * 密码算法 ECB 56  (DES/ECB/PKCS5Padding) (DES/ECB/NoPadding)
     */
    private static final String CIPHER_ALGORITHM_ECB = "DES/ECB/PKCS5Padding"; // "算法/模式/补码方式"
    /**
     * 密码算法 CBC 56  (DES/CBC/PKCS5Padding) (DES/CBC/NoPadding)
     */
    private static final String CIPHER_ALGORITHM_CBC = "DES/CBC/PKCS5Padding"; // "算法/模式/补码方式"

    /**
     * 获取加密对象
     *
     * @param mode 模式：加密模式 / 解密模式
     * @param key  密钥
     * @param iv   偏移量
     * @return 加密对象
     * @throws Exception 异常
     */
    private static Cipher getCipher(int mode, String key, String iv) {
        try {
            // 创建加密规则
            final SecretKeySpec secretKeySpec = new SecretKeySpec(CodecUtil.decodeBase64(key), KEY_ALGORITHM_DES);

            // 创建加密对象
            Cipher cipher;
            if (StringUtil.isBlank(iv)) {
                cipher = Cipher.getInstance(CIPHER_ALGORITHM_ECB); // "算法/模式/补码方式"
                cipher.init(mode, secretKeySpec); // 初始化加密模式和算法
            } else {
                cipher = Cipher.getInstance(CIPHER_ALGORITHM_CBC); // "算法/模式/补码方式"
                IvParameterSpec ivParameterSpec = new IvParameterSpec(CodecUtil.decodeBase64(iv)); // 创建 iv 向量， 是使用到 CBC 加密模式
                cipher.init(mode, secretKeySpec, ivParameterSpec); // 初始化加密模式和算法
            }

            return cipher;
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 生成AES的秘钥，秘钥进行了Base64编码的字符串
     *
     * @return String 对生成的秘钥进行了Base64编码的字符串
     */
    public static String keyGenerate() {
        try {
            KeyGenerator keyGenerator = KeyGenerator.getInstance(KEY_ALGORITHM_DES);
            keyGenerator.init(new SecureRandom());
            SecretKey secretKey = keyGenerator.generateKey();
            byte[] keyBytes = secretKey.getEncoded();
            return CodecUtil.encodeBase64(keyBytes);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 生成随机的密钥 key
     *
     * @return 随机密钥 key
     * @throws Exception 异常
     */
    public static String generateKey() {
        return keyGenerate();
    }

    /**
     * 生成随机的偏移量 iv
     *
     * @return 随机的偏移量 iv
     * @throws Exception 异常
     */
    public static String generateIv() {
        return keyGenerate();
    }


    /**
     * 获取随机密钥
     *
     * @return 随机密钥
     */
    public static String getRandomKey() {
        final String randomCode = StringUtil.getRandomCode(8);
        return CodecUtil.encodeBase64(randomCode);
    }

    /**
     * 获取随机Iv
     *
     * @return 随机密钥
     */
    public static String getRandomIv() {
        final String randomCode = StringUtil.getRandomCode(8);
        return CodecUtil.encodeBase64(randomCode);
    }
}
