package cn.net.wanmo.common.crypto.gm;

import cn.net.wanmo.common.charset.CharsetUtil;
import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.crypto.pojo.KeyPairBase64;
import cn.net.wanmo.common.crypto.pojo.KeyPairFiles;
import cn.net.wanmo.common.util.Exceptions;
import cn.net.wanmo.common.util.FileUtil;
import cn.net.wanmo.common.util.StringUtil;
import org.bouncycastle.asn1.gm.GMObjectIdentifiers;
import org.bouncycastle.crypto.InvalidCipherTextException;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jce.spec.ECParameterSpec;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.security.*;
import java.security.spec.ECGenParameterSpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

public class Sm2Util {
    /** 密钥算法 EC */
    public static final String KEY_ALGORITHM = "EC";
    /** 密钥算法 提供者 */
    public static final Provider provider = new BouncyCastleProvider();

    static {
        Security.addProvider(provider);
    }

    /**
     * 获取密钥对
     *
     * @return 密钥对
     */
    public static KeyPair getKeyPair() {
        try {
            SecureRandom secureRandom = new SecureRandom();
            // 获取SM2椭圆曲线的参数
            final ECGenParameterSpec sm2Spec = new ECGenParameterSpec("sm2p256v1");
            // 获取一个椭圆曲线类型的密钥对生成器
            final KeyPairGenerator kpg = KeyPairGenerator.getInstance(KEY_ALGORITHM, provider);
            // 使用SM2参数初始化生成器
            kpg.initialize(sm2Spec);
            // 使用SM2的算法区域初始化密钥生成器
            kpg.initialize(sm2Spec, secureRandom);
            // 获取密钥对
            KeyPair keyPair = kpg.generateKeyPair();
            // 返回 KeyPair
            return keyPair;
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 获取公钥和私钥的Base64
     */
    public static KeyPairBase64 getKeyPairBase64() {
        final KeyPair keyPair = getKeyPair();

        // 获取私钥和公钥的字节数组
        final byte[] privateEncoded = keyPair.getPrivate().getEncoded();
        final byte[] publicEncoded = keyPair.getPublic().getEncoded();

        // 使用base64 转码
        final String privateEncodedBase64 = CodecUtil.encodeBase64(privateEncoded);
        final String publicEncodedBase64 = CodecUtil.encodeBase64(publicEncoded);

        return new KeyPairBase64(publicEncodedBase64, privateEncodedBase64);
    }

    /**
     * 保存公钥和私钥到文件
     *
     * @param pubFilePath 公钥路径
     * @param priFilePath 私有路径
     */
    public static KeyPairFiles getKeyPairFiles(String pubFilePath, String priFilePath) {
        try {
            final KeyPairBase64 keyPairBase64 = getKeyPairBase64();
            final KeyPairFiles keyPairFiles = new KeyPairFiles(pubFilePath, priFilePath);

            FileUtil.writeStringToFile(keyPairFiles.getPublicKey(), keyPairBase64.getPublicKey(), CharsetUtil.DEFAULT);
            FileUtil.writeStringToFile(keyPairFiles.getPublicKey(), keyPairBase64.getPrivateKey(), CharsetUtil.DEFAULT);

            return keyPairFiles;
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }



    /**
     * 从 Base64 获取公钥
     *
     * @param publicKeyBase64 公钥的Base64
     * @return 公钥
     */
    public static PublicKey getPublicKeyByBase64(String publicKeyBase64) {
        try {
            // 创建 key 工厂
            final KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM, provider);
            // 创建公钥 key 的规则
            final X509EncodedKeySpec keySpec = new X509EncodedKeySpec(CodecUtil.decodeBase64(publicKeyBase64));
            // 返回私钥对象
            return keyFactory.generatePublic(keySpec);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 从 Base64 获取私钥
     *
     * @param privateKeyBase64 私钥的Base64
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByBase64(String privateKeyBase64) {
        try {
            // 创建 key 工厂
            final KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM, provider);
            // 创建私钥 key 的规则
            final PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(CodecUtil.decodeBase64(privateKeyBase64));
            // 返回私钥对象
            return keyFactory.generatePrivate(keySpec);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }
    /**
     * 从文件读取私钥
     *
     * @param priFilePath 私钥的路径
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByFilePath(String priFilePath) {
        return getPrivateKeyByFilePath(priFilePath, CharsetUtil.DEFAULT);
    }

    /**
     * 从文件读取私钥
     *
     * @param priFilePath 私钥的路径
     * @param encoding 字符集
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByFilePath(String priFilePath, Charset encoding) {
        return getPrivateKeyByFile(FileUtil.getFile(priFilePath), encoding);
    }

    /**
     * 从文件读取私钥
     *
     * @param priFile 私钥的路径
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByFile(File priFile) {
        return getPrivateKeyByFile(priFile, CharsetUtil.DEFAULT);
    }

    /**
     * 从文件读取私钥
     *
     * @param priFile 私钥的路径
     * @param encoding 字符集
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByFile(File priFile, Charset encoding) {
        try {
            // 获取 key 字符串
            final String privateKeyBase64 = FileUtil.readFileToString(priFile, encoding);
            // 返回私钥对象
            return getPrivateKeyByBase64(privateKeyBase64);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 从文件读取公钥
     *
     * @param pubFilePath 公钥的路径
     * @return 公钥
     */
    public static PublicKey getPublicKeyByFilePath(String pubFilePath) {
        return getPublicKeyByFilePath(pubFilePath, CharsetUtil.DEFAULT);
    }

    /**
     * 从文件读取公钥
     *
     * @param pubFilePath 公钥的路径
     * @param encoding 字符集
     * @return 公钥
     */
    public static PublicKey getPublicKeyByFilePath(String pubFilePath, Charset encoding) {
        return getPublicKeyByFile(FileUtil.getFile(pubFilePath), encoding);
    }

    /**
     * 从文件读取公钥
     *
     * @param pubFile 公钥的路径
     * @return 公钥
     */
    public static PublicKey getPublicKeyByFile(File pubFile) {
        return getPublicKeyByFile(pubFile, CharsetUtil.DEFAULT);
    }

    /**
     * 从文件读取公钥
     *
     * @param pubFile 公钥的路径
     * @param encoding 字符集
     * @return 公钥
     */
    public static PublicKey getPublicKeyByFile(File pubFile, Charset encoding) {
        try {
            // 获取 key 字符串
            final String publicKeyBase64 = FileUtil.readFileToString(pubFile, encoding);
            // 返回私钥对象
            return getPublicKeyByBase64(publicKeyBase64);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 根据publicKey对原始数据data，使用SM2加密
     *
     * @param data 原文
     * @param publicKeyBase64 公钥
     * @return 加密数据
     */
    public static String encrypt(String data, String publicKeyBase64) {
        return encrypt(data, publicKeyBase64, CharsetUtil.DEFAULT);
    }

    /**
     * 根据publicKey对原始数据data，使用SM2加密
     *
     * @param data 原文
     * @param publicKeyBase64 公钥
     * @param charset 字符集
     * @return 加密数据
     */
    public static String encrypt(String data, String publicKeyBase64, Charset charset) {
        final PublicKey publicKey = getPublicKeyByBase64(publicKeyBase64);
        return encrypt(data, publicKey, charset);
    }

    /**
     * 根据 publicKey 对原始数据 data，使用SM2加密
     *
     * @param data 原文
     * @param publicKey 公钥
     * @return 加密数据
     */
    public static String encrypt(String data, PublicKey publicKey) {
        return encrypt(data, publicKey, CharsetUtil.DEFAULT);
    }

    /**
     * 根据 publicKey 对原始数据 data，使用SM2加密
     *
     * @param data 原文
     * @param publicKey 公钥
     * @param charset 字符集
     * @return 加密数据
     */
    public static String encrypt(String data, PublicKey publicKey, Charset charset) {
        try {
            SM2Engine sm2Engine = new SM2Engine();

            ECPublicKeyParameters publicKeyParameters = null;
            if (publicKey instanceof BCECPublicKey) {
                BCECPublicKey sm2PubK = (BCECPublicKey) publicKey;
                ECParameterSpec spec = sm2PubK.getParameters();
                ECDomainParameters domainParameters = new ECDomainParameters(spec.getCurve(), spec.getG(), spec.getN());
                publicKeyParameters = new ECPublicKeyParameters(sm2PubK.getQ(), domainParameters);
            }
            sm2Engine.init(true, new ParametersWithRandom(publicKeyParameters, new SecureRandom()));

            final byte[] dataBytes = data.getBytes(charset);
            byte[] encryptBytes = sm2Engine.processBlock(dataBytes, 0, dataBytes.length);

            // Base64 编码
            return CodecUtil.encodeBase64(encryptBytes);
        } catch (InvalidCipherTextException e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 根据 privateKey 对加密数据 ciphertext，使用SM2解密
     *
     * @param cipherText 密文
     * @param privateKeyBase64 私钥
     * @return 原文
     */
    public static String decrypt(String cipherText,  String privateKeyBase64) {
        return decrypt(cipherText, privateKeyBase64, CharsetUtil.DEFAULT);
    }

    /**
     * 根据 privateKey 对加密数据 ciphertext，使用SM2解密
     *
     * @param cipherText 密文
     * @param privateKeyBase64 私钥
     * @param charset 字符集
     * @return 原文
     */
    public static String decrypt(String cipherText,  String privateKeyBase64, Charset charset) {
        final PrivateKey privateKey = getPrivateKeyByBase64(privateKeyBase64);
        return decrypt(cipherText, privateKey, charset);
    }

    /**
     * 根据 privateKey 对加密数据 cipherText，使用SM2解密
     *
     * @param cipherText 密文
     * @param privateKey 私钥
     * @return 原文
     */
    public static String decrypt(String cipherText,  PrivateKey privateKey) {
        return decrypt(cipherText, privateKey, CharsetUtil.DEFAULT);
    }

    /**
     * 根据 privateKey 对加密数据 cipherText，使用SM2解密
     *
     * @param cipherText 密文
     * @param privateKey 私钥
     * @param charset 字符集
     * @return 原文
     */
    public static String decrypt(String cipherText,  PrivateKey privateKey, Charset charset) {
        try {
            SM2Engine sm2Engine = new SM2Engine();

            ECPrivateKeyParameters privateKeyParameters = null;
            if (privateKey instanceof BCECPrivateKey) {
                BCECPrivateKey sm2PriK = (BCECPrivateKey) privateKey;
                ECParameterSpec spec = sm2PriK.getParameters();
                ECDomainParameters domainParameters = new ECDomainParameters(spec.getCurve(), spec.getG(), spec.getN());
                privateKeyParameters = new ECPrivateKeyParameters(sm2PriK.getD(), domainParameters);
            }
            sm2Engine.init(false, privateKeyParameters);

            final byte[] ciphertextBytes = CodecUtil.decodeBase64(cipherText);
            byte[] decryptBytes = sm2Engine.processBlock(ciphertextBytes, 0, ciphertextBytes.length);

            // 密钥解密，使用默认字符集，返回字符串
            return StringUtil.toEncodedString(decryptBytes, charset);
        } catch (InvalidCipherTextException e) {
            throw Exceptions.unchecked(e);
        }
    }

    /** 签名验签算法 */
    public static final String SM3_WITH_SM2 = "SM3withSm2";

    /**
     * 数据签名
     * @param privateKeyBase64 私钥
     * @param data 数据
     * @return 签名值
     */
    public static String getSignatureForSm3(String privateKeyBase64, String data) {
        final PrivateKey privateKey = getPrivateKeyByBase64(privateKeyBase64);
        return getSignature(SM3_WITH_SM2, privateKey, data);
    }

    /**
     * 数据签名
     * @param privateKey 私钥
     * @param data 数据
     * @return 签名值
     */
    public static String getSignatureForSm3(PrivateKey privateKey, String data) {
        return getSignature(SM3_WITH_SM2, privateKey, data);
    }

    /**
     * 私钥数据签名
     * @param algorithm 签名算法： SM3withSm2
     * @param privateKey 私钥
     * @param data 数据
     * @return 签名值
     */
    public static String getSignature(String algorithm, PrivateKey privateKey, String data) {
        try {
            // 获取签名对象
            final Signature signature = Signature.getInstance(algorithm, new BouncyCastleProvider());
            // 签名需要使用私钥，使用私钥 初始化签名实例
            signature.initSign(privateKey);
            // 写入签名原文到算法中
            signature.update(data.getBytes());
            // 计算签名值
            final byte[] signatureValue = signature.sign();
            // 使用 16进制 进行编码
            return CodecUtil.encodeHex(signatureValue);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 公钥数据验签
     * @param publicKeyBase64 公钥
     * @param data 数据
     * @param signatureData 签名值
     * @return 验签结果
     */
    public static boolean verifySignatureForSm3(String publicKeyBase64, String data, String signatureData) {
        final PublicKey publicKey = getPublicKeyByBase64(publicKeyBase64);
        return verifySignature(SM3_WITH_SM2, publicKey, data, signatureData);
    }

    /**
     * 公钥数据验签
     * @param publicKey 公钥
     * @param data 数据
     * @param signatureData 签名值
     * @return 验签结果
     */
    public static boolean verifySignatureForSm3(PublicKey publicKey, String data, String signatureData) {
        return verifySignature(SM3_WITH_SM2, publicKey, data, signatureData);
    }

    /**
     * 公钥数据验签
     * @param algorithm 签名算法： sha256withrsa
     * @param publicKey 公钥
     * @param data 数据
     * @param signatureData 签名值
     * @return 验签结果
     */
    public static boolean verifySignature(String algorithm, PublicKey publicKey, String data, String signatureData) {
        try {
            // 获取签名对象
            final Signature signature = Signature.getInstance(algorithm, new BouncyCastleProvider());
            // 签名需要使用公钥，使用公钥 初始化签名实例
            signature.initVerify(publicKey);
            // 写入待验签的签名原文到算法中
            signature.update(data.getBytes());
            // 校验数据
            return signature.verify(CodecUtil.decodeHex(signatureData));
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 私钥签名
     *
     * @param data 原文
     * @param privateKeyBase64 私钥
     * @return 签名值
     */
    public static String signWithSm3(String privateKeyBase64, String data) {
        final PrivateKey privateKey = getPrivateKeyByBase64(privateKeyBase64);
        return signWithSm3(privateKey, data);
    }

    /**
     * 私钥签名
     *
     * @param data 原文
     * @param privateKey 私钥
     * @return 签名值
     */
    public static String signWithSm3(PrivateKey privateKey, String data) {
        try {
            // 获取签名对象
            Signature sig = Signature.getInstance(GMObjectIdentifiers.sm2sign_with_sm3.toString(), BouncyCastleProvider.PROVIDER_NAME);
            // 签名需要使用私钥，使用私钥 初始化签名实例
            sig.initSign(privateKey);
            // 写入签名原文到算法中
            sig.update(data.getBytes());
            // 计算签名值
            final byte[] signatureValue = sig.sign();
            // 使用 16进制 进行编码
            return CodecUtil.encodeHex(signatureValue);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 公钥验签
     *
     * @param data 原文
     * @param publicKeyBase64 公钥
     * @param signatureData 签名值
     * @return 验签结果
     */
    public static boolean verifyWithSm3(String publicKeyBase64, String data, String signatureData) {
        final PublicKey publicKey = getPublicKeyByBase64(publicKeyBase64);
        return verifyWithSm3(publicKey, data, signatureData);
    }

    /**
     * 公钥验签
     *
     * @param data 原文
     * @param publicKey 公钥
     * @param signatureData 签名值
     * @return 验签结果
     */
    public static boolean verifyWithSm3(PublicKey publicKey, String data, String signatureData) {
        try {
            // 获取签名对象
            Signature sig = Signature.getInstance(GMObjectIdentifiers.sm2sign_with_sm3.toString(), BouncyCastleProvider.PROVIDER_NAME);
            // 签名需要使用公钥，使用公钥 初始化签名实例
            sig.initVerify(publicKey);
            // 写入待验签的签名原文到算法中
            sig.update(data.getBytes());
            // 校验数据
            return sig.verify(CodecUtil.decodeHex(signatureData));
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }
}
