package cn.net.wanmo.common.crypto.gm;


import cn.net.wanmo.common.charset.CharsetUtil;
import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.util.Exceptions;
import cn.net.wanmo.common.util.StringUtil;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.security.Key;
import java.security.SecureRandom;
import java.security.Security;

public class Sm4Util {
    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    public static final String ALGORITHM_NAME = "SM4";
    public static final String DEFAULT_KEY = "wan_mo_seed";

    /** 128 -32位16进制；256 -64位16进制 */
    public static final int DEFAULT_KEY_SIZE = 128;

    /**
     * 生产 key，长度默认 128
     * @return key
     */
    public static String generateKey() {
        return generateKey(DEFAULT_KEY, DEFAULT_KEY_SIZE);
    }

    /**
     * 生产 key，长度默认 128
     * @param seed 种子
     * @return key
     */
    public static String generateKey(String seed)  {
        return generateKey(seed, DEFAULT_KEY_SIZE);
    }

    /**
     * 生产 key
     * @param seed 种子
     * @param keySize 长度
     * @return key
     */
    public static String generateKey(String seed, int keySize)  {
        try {
            KeyGenerator kg = KeyGenerator.getInstance(ALGORITHM_NAME, BouncyCastleProvider.PROVIDER_NAME);
            SecureRandom random = SecureRandom.getInstance("SHA1PRNG");

            if (StringUtil.isNotBlank(seed)) {
                random.setSeed(seed.getBytes());
            }

            kg.init(keySize, random);

            final byte[] encoded = kg.generateKey().getEncoded();
            return CodecUtil.encodeBase64(encoded);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 生产 16位的 iv
     * @return iv
     */
    public static String generateIv() {
        final String randomCode = StringUtil.getRandomCode(16);
        return CodecUtil.encodeBase64(randomCode);
    }

    /**
     * 加密
     * @param key 密钥
     * @param iv IV
     * @param data 原文
     * @return 密文
     */
    public static String encryptCBC(String key, String iv, String data) {
        return encrypt("SM4/CBC/PKCS5PADDING", key, iv, data);
    }

    /**
     * 解密
     * @param key 密钥
     * @param iv IV
     * @param data 加密数据
     * @return 原文
     */
    public static String decryptCBC(String key, String iv, String data) {
        return decrypt("SM4/CBC/PKCS5PADDING", key, iv, data);
    }

    /**
     * 加密
     * @param key 密钥
     * @param data 原文
     * @return 密文
     */
    public static String encryptECB(String key, String data) {
        return encrypt("SM4/ECB/PKCS5PADDING", key, "", data);
    }

    /**
     * 解密
     * @param key 密钥
     * @param data 加密数据
     * @return 原文
     */
    public static String decryptECB(String key, String data) {
        return decrypt("SM4/ECB/PKCS5PADDING", key, "", data);
    }

    /**
     * 加密
     * @param algorithmName 算法
     * @param key 密钥
     * @param iv IV
     * @param data 原文
     * @return 密文
     */
    public static String encrypt(String algorithmName, String key, String iv, String data) {
        byte[] keyBytes = CodecUtil.decodeBase64(key);
        byte[] ivBytes = CodecUtil.decodeBase64(iv);
        byte[] dataBytes = data.getBytes(CharsetUtil.DEFAULT);

        final byte[] encryptBytes = sm4core(algorithmName, Cipher.ENCRYPT_MODE, keyBytes, ivBytes, dataBytes);
        return CodecUtil.encodeBase64(encryptBytes);
    }


    /**
     * 解密
     * @param algorithmName 算法
     * @param key 密钥
     * @param iv IV
     * @param data 加密数据
     * @return 原文
     */
    public static String decrypt(String algorithmName, String key, String iv, String data) {
        byte[] keyBytes = CodecUtil.decodeBase64(key);
        byte[] ivBytes = CodecUtil.decodeBase64(iv);
        byte[] dataBytes = CodecUtil.decodeBase64(data);

        final byte[] decryptBytes = sm4core(algorithmName, Cipher.DECRYPT_MODE, keyBytes, ivBytes, dataBytes);
        return StringUtil.toEncodedString(decryptBytes, CharsetUtil.DEFAULT);
    }

    /**
     * 加解密核心方法
     * @param algorithmName 算法
     * @param type 加解密类型（加密：Cipher.ENCRYPT_MODE，解密：Cipher.DECRYPT_MODE）
     * @param key 密钥
     * @param iv IV
     * @param data 原文或密文
     * @return 密文或原文
     */
    private static byte[] sm4core(String algorithmName, int type, byte[] key, byte[] iv, byte[] data) {
        try {
            Cipher cipher = Cipher.getInstance(algorithmName, BouncyCastleProvider.PROVIDER_NAME);
            Key sm4Key = new SecretKeySpec(key, ALGORITHM_NAME);

            if (algorithmName.contains("/ECB/")) {
                cipher.init(type, sm4Key);
            } else {
                IvParameterSpec ivParameterSpec = new IvParameterSpec(iv);
                cipher.init(type, sm4Key, ivParameterSpec);
            }

            return cipher.doFinal(data);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

}
