package cn.net.wanmo.common.crypto.rsa;

import cn.net.wanmo.common.charset.CharsetUtil;
import cn.net.wanmo.common.codec.CodecUtil;
import cn.net.wanmo.common.crypto.pojo.KeyPairBase64;
import cn.net.wanmo.common.crypto.pojo.KeyPairFiles;
import cn.net.wanmo.common.util.Exceptions;
import cn.net.wanmo.common.util.FileUtil;
import cn.net.wanmo.common.util.StringUtil;

import javax.crypto.Cipher;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;

/**
 * RSA/ECB/PKCS1Padding
 * RSA/ECB/0AEPWithSHA-1AndMGF1Padding
 * RSA/ECB/0AEPWithSHA-256AndMGF1Padding
 */
public class RsaUtil {

    /** 密钥算法 RSA */
    public static final String KEY_ALGORITHM = "RSA";

    /**
     * 获取密钥对
     *
     * @return 密钥对
     */
    public static KeyPair getKeyPair() {
        try {
            final KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(KEY_ALGORITHM);
            keyPairGenerator.initialize(2048);
            return keyPairGenerator.generateKeyPair();
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }


    /**
     * 获取公钥和私钥的Base64
     */
    public static KeyPairBase64 getKeyPairBase64() {
        final KeyPair keyPair = getKeyPair();

        // 获取私钥和公钥的字节数组
        final byte[] privateEncoded = keyPair.getPrivate().getEncoded();
        final byte[] publicEncoded = keyPair.getPublic().getEncoded();

        // 使用base64 转码
        final String privateEncodedBase64 = CodecUtil.encodeBase64(privateEncoded);
        final String publicEncodedBase64 = CodecUtil.encodeBase64(publicEncoded);

        return new KeyPairBase64(publicEncodedBase64, privateEncodedBase64);
    }

    /**
     * 保存公钥和私钥到文件
     *
     * @param pubFilePath 公钥路径
     * @param priFilePath 私有路径
     */
    public static KeyPairFiles getKeyPairFiles(String pubFilePath, String priFilePath) {
        return getKeyPairFiles(pubFilePath, priFilePath, CharsetUtil.DEFAULT);
    }

    /**
     * 保存公钥和私钥到文件
     *
     * @param pubFilePath 公钥路径
     * @param priFilePath 私有路径
     * @param charset    字符集
     */
    public static KeyPairFiles getKeyPairFiles(String pubFilePath, String priFilePath, Charset charset) {
        try {
            final KeyPairBase64 keyPairBase64 = getKeyPairBase64();
            final KeyPairFiles keyPairFiles = new KeyPairFiles(pubFilePath, priFilePath);

            FileUtil.writeStringToFile(keyPairFiles.getPublicKey(), keyPairBase64.getPublicKey(), charset);
            FileUtil.writeStringToFile(keyPairFiles.getPrivateKey(), keyPairBase64.getPrivateKey(), charset);

            return keyPairFiles;
        } catch (IOException e) {
            throw Exceptions.unchecked(e);
        }
    }



    /**
     * 从 Base64 获取公钥
     *
     * @param publicKeyBase64 公钥的Base64
     * @return 公钥
     */
    public static PublicKey getPublicKeyByBase64(String publicKeyBase64) {
        try {
            // 创建 key 工厂
            final KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            // 创建公钥 key 的规则
            final X509EncodedKeySpec keySpec = new X509EncodedKeySpec(CodecUtil.decodeBase64(publicKeyBase64));
            // 返回私钥对象
            return keyFactory.generatePublic(keySpec);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 从 Base64 获取私钥
     *
     * @param privateKeyBase64 私钥的Base64
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByBase64(String privateKeyBase64) {
        try {
            // 创建 key 工厂
            final KeyFactory keyFactory = KeyFactory.getInstance(KEY_ALGORITHM);
            // 创建私钥 key 的规则
            final PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(CodecUtil.decodeBase64(privateKeyBase64));
            // 返回私钥对象
            return keyFactory.generatePrivate(keySpec);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 从文件读取私钥
     *
     * @param priFilePath 私钥的路径
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByFilePath(String priFilePath) {
        return getPrivateKeyByFilePath(priFilePath, CharsetUtil.DEFAULT);
    }
    /**
     * 从文件读取私钥
     *
     * @param priFilePath 私钥的路径
     * @param charset    字符集
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByFilePath(String priFilePath, Charset charset) {
        return getPrivateKeyByFile(FileUtil.getFile(priFilePath), charset);
    }

    /**
     * 从文件读取私钥
     *
     * @param priFile 私钥的路径
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByFile(File priFile) {
        return getPrivateKeyByFile(priFile, CharsetUtil.DEFAULT);
    }
    /**
     * 从文件读取私钥
     *
     * @param priFile 私钥的路径
     * @param charset    字符集
     * @return 私钥
     */
    public static PrivateKey getPrivateKeyByFile(File priFile, Charset charset) {
        try {
            // 获取 key 字符串
            final String privateKeyBase64 = FileUtil.readFileToString(priFile, charset);
            // 返回私钥对象
            return getPrivateKeyByBase64(privateKeyBase64);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 从文件读取公钥
     *
     * @param pubFilePath 公钥的路径
     * @return 公钥
     */
    public static PublicKey getPublicKeyByFilePath(String pubFilePath) {
        return getPublicKeyByFilePath(pubFilePath, CharsetUtil.DEFAULT);
    }

    /**
     * 从文件读取公钥
     *
     * @param pubFilePath 公钥的路径
     * @param charset    字符集
     * @return 公钥
     */
    public static PublicKey getPublicKeyByFilePath(String pubFilePath, Charset charset) {
        return getPublicKeyByFile(FileUtil.getFile(pubFilePath), charset);
    }

    /**
     * 从文件读取公钥
     *
     * @param pubFile 公钥的路径
     * @return 公钥
     */
    public static PublicKey getPublicKeyByFile(File pubFile) {
        return getPublicKeyByFile(pubFile, CharsetUtil.DEFAULT);
    }

    /**
     * 从文件读取公钥
     *
     * @param pubFile 公钥的路径
     * @param charset    字符集
     * @return 公钥
     */
    public static PublicKey getPublicKeyByFile(File pubFile, Charset charset) {
        try {
            // 获取 key 字符串
            final String publicKeyBase64 = FileUtil.readFileToString(pubFile, charset);
            // 返回私钥对象
            return getPublicKeyByBase64(publicKeyBase64);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }


    /**
     * 获取加解密对象
     *
     * @param mode 加解密模式
     * @param key  密钥（公钥或私钥）
     * @return 加解密对象
     */
    public static Cipher getCipher(int mode, Key key) {
        try {
            // 创建加密对象
            final Cipher cipher = Cipher.getInstance(KEY_ALGORITHM);
            // 初始化加密对象，私钥
            cipher.init(mode, key);
            // 返回加解密对象
            return cipher;
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 私钥加密
     *
     * @param data 原文
     * @param priKeyBase64  私钥 Base64
     * @return 密文
     */
    public static String encryptByPri(String data, String priKeyBase64) {
        return encryptByPri(data, priKeyBase64, CharsetUtil.DEFAULT);
    }

    /**
     * 私钥加密
     *
     * @param data 原文
     * @param priKeyBase64  私钥 Base64
     * @param charset    字符集
     * @return 密文
     */
    public static String encryptByPri(String data, String priKeyBase64, Charset charset) {
        // 获取私钥
        final PrivateKey privateKey = getPrivateKeyByBase64(priKeyBase64);
        // 私钥加密
        return encrypt(data, privateKey, charset);
    }

    /**
     * 公钥加密
     *
     * @param data 原文
     * @param pubKeyBase64  公钥 Base64
     * @return 密文
     */
    public static String encryptByPub(String data, String pubKeyBase64) {
        return encryptByPub(data, pubKeyBase64, CharsetUtil.DEFAULT);
    }

    /**
     * 公钥加密
     *
     * @param data 原文
     * @param pubKeyBase64  公钥 Base64
     * @param charset    字符集
     * @return 密文
     */
    public static String encryptByPub(String data, String pubKeyBase64, Charset charset) {
        // 获取公钥
        final PublicKey publicKey = getPublicKeyByBase64(pubKeyBase64);
        // 公钥加密
        return encrypt(data, publicKey, charset);
    }
    /**
     * 加密
     *
     * @param data 原文
     * @param key  密钥
     * @return 密文
     */
    public static String encrypt(String data, Key key) {
        return encrypt(data, key, CharsetUtil.DEFAULT);
    }

    /**
     * 加密
     *
     * @param data 原文
     * @param key  密钥
     * @param charset    字符集
     * @return 密文
     */
    public static String encrypt(String data, Key key, Charset charset) {
        try {
            // 创建加密对象
            final Cipher cipherPrivate = getCipher(Cipher.ENCRYPT_MODE, key);
            // 密钥加密，使用默认字符集获取原文的字节数组
            final byte[] bytes = cipherPrivate.doFinal(data.getBytes(charset));
            // Base64 编码
            return CodecUtil.encodeBase64(bytes);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 私钥解密
     *
     * @param cipherText 密文
     * @param priKeyBase64        私钥 Base64
     * @return 原文
     */
    public static String decryptByPri(String cipherText, String priKeyBase64) {
        return decryptByPri(cipherText, priKeyBase64, CharsetUtil.DEFAULT);
    }

    /**
     * 私钥解密
     *
     * @param cipherText 密文
     * @param priKeyBase64        私钥 Base64
     * @param charset    字符集
     * @return 原文
     */
    public static String decryptByPri(String cipherText, String priKeyBase64, Charset charset) {
        // 获取私钥
        final PrivateKey privateKey = getPrivateKeyByBase64(priKeyBase64);
        // 私钥解密
        return decrypt(cipherText, privateKey, charset);
    }

    /**
     * 公钥解密
     *
     * @param cipherText 密文
     * @param pubKeyBase64        公钥 Base64
     * @return 原文
     */
    public static String decryptByPub(String cipherText, String pubKeyBase64) {
        return decryptByPub(cipherText, pubKeyBase64, CharsetUtil.DEFAULT);
    }

    /**
     * 公钥解密
     *
     * @param cipherText 密文
     * @param pubKeyBase64        公钥 Base64
     * @param charset    字符集
     * @return 原文
     */
    public static String decryptByPub(String cipherText, String pubKeyBase64, Charset charset) {
        // 获取公钥
        final PublicKey publicKey = getPublicKeyByBase64(pubKeyBase64);
        // 公钥解密
        return decrypt(cipherText, publicKey, charset);
    }

    /**
     * 解密
     *
     * @param cipherText 密文
     * @param key        密钥
     * @return 原文
     */
    public static String decrypt(String cipherText, Key key) {
        return decrypt(cipherText, key, CharsetUtil.DEFAULT);
    }

    /**
     * 解密
     *
     * @param cipherText 密文
     * @param key        密钥
     * @param charset    字符集
     * @return 原文
     */
    public static String decrypt(String cipherText, Key key, Charset charset) {
        try {
            // 创建解密对象
            final Cipher cipherPublic = getCipher(Cipher.DECRYPT_MODE, key);
            // 解密
            final byte[] decryptBytes = cipherPublic.doFinal(CodecUtil.decodeBase64(cipherText));
            // 密钥解密，使用默认字符集，返回字符串
            return StringUtil.toEncodedString(decryptBytes, charset);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }


    /** 签名算法： sha256withrsa */
    public static final String SHA256_WITH_RSA = "sha256withrsa";

    /**
     * 私钥数据签名
     * @param privateKeyBase64 私钥 Base64
     * @param data 数据
     * @return 签名值
     */
    public static String getSignatureForSha256(String privateKeyBase64, String data) {
        final PrivateKey privateKey = getPrivateKeyByBase64(privateKeyBase64);
        return getSignatureForSha256(privateKey, data);
    }

    /**
     * 私钥数据签名
     * @param privateKey 私钥
     * @param data 数据
     * @return 签名值
     */
    public static String getSignatureForSha256(PrivateKey privateKey, String data) {
        return getSignature(SHA256_WITH_RSA, privateKey, data);
    }

    /**
     * 私钥数据签名
     * @param algorithm 签名算法： sha256withrsa
     * @param privateKey 私钥
     * @param data 数据
     * @return 签名值
     */
    public static String getSignature(String algorithm, PrivateKey privateKey, String data) {
        try {
            // 获取签名对象
            final Signature signature = Signature.getInstance(algorithm);
            // 初始化签名
            signature.initSign(privateKey);
            // 传入原文
            signature.update(data.getBytes());
            // 开始签名
            final byte[] sign = signature.sign();
            // 使用 16进制 进行编码
            return CodecUtil.encodeHex(sign);
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }

    /**
     * 公钥数据验签
     * @param publicKeyBase64 公钥 Base64
     * @param data 数据
     * @param signatureData 签名值
     * @return 验签结果
     */
    public static boolean verifySignatureForSha256(String publicKeyBase64, String data, String signatureData) {
        final PublicKey publicKey = getPublicKeyByBase64(publicKeyBase64);
        return verifySignatureForSha256(publicKey, data, signatureData);
    }

    /**
     * 公钥数据验签
     * @param publicKey 公钥
     * @param data 数据
     * @param signatureData 签名值
     * @return 验签结果
     */
    public static boolean verifySignatureForSha256(PublicKey publicKey, String data, String signatureData) {
        return verifySignature(SHA256_WITH_RSA, publicKey, data, signatureData);
    }

    /**
     * 公钥数据验签
     * @param algorithm 签名算法： sha256withrsa
     * @param publicKey 公钥
     * @param data 数据
     * @param signatureData 签名值
     * @return 验签结果
     */
    public static boolean verifySignature(String algorithm, PublicKey publicKey, String data, String signatureData) {
        try {
            // 获取签名对象
            final Signature signature = Signature.getInstance(algorithm);
            // 初始化签名
            signature.initVerify(publicKey);
            // 传入原文
            signature.update(data.getBytes());
            // 校验数据
            return signature.verify(CodecUtil.decodeHex(signatureData));
        } catch (Exception e) {
            throw Exceptions.unchecked(e);
        }
    }
}
