package cn.net.wanmo.plugin.spring.util.http;

import cn.net.wanmo.common.util.IdGen;
import cn.net.wanmo.plugin.spring.util.SpringUtil;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.http.*;
import org.springframework.util.Assert;
import org.springframework.web.client.RestTemplate;

import java.io.*;

public class RestTemplateUtil {
    private static final Logger logger = LoggerFactory.getLogger(RestTemplateUtil.class);
    private static final RestTemplate restTemplate = SpringUtil.getBean(RestTemplate.class);

    /**
     * 通用指定返回对象类型
     * @param url 请求地址
     * @param responseType 返回的类型
     * @param <T> 方法内部泛型
     * @return 获取 Http 请求的结果
     */
    public static  <T> T getForObject(String url, Class<T> responseType) {
        return restTemplate.getForObject(url, responseType);
    }

    /**
     * 通用指定返回对象类型
     * @param url 请求地址
     * @param responseType 返回的类型
     * @param <T> 方法内部泛型
     * @return 获取 Http 请求的结果
     */
    public static  <T> T getForEntity(String url, Class<T> responseType) {
        return restTemplate.getForEntity(url, responseType).getBody();
    }

    /**
     * GET 请求，支持 Https
     * @param url 请求地址
     * @return 返回响应字符串
     */
    public static String getForString(String url) {
        assertRestTemplate();

        String response = restTemplate.getForObject(url, String.class);
        logger.debug("response = {}", response);

        return response;
    }

    /**
     * 使用JSON参数发送Post请求，支持 Https
     * @param url 请求地址
     * @param params 参数: code=abc123&token=abc
     * @return  返回响应字符串
     */
    public static String postForString(String url, String params) {
        assertRestTemplate();

        HttpHeaders headers = new HttpHeaders();
        HttpEntity<String> request = new HttpEntity<>(params, headers);

        String response = restTemplate.postForObject(url, request, String.class);
        logger.debug("response = {}", response);

        return response;
    }

    /**
     * 使用JSON参数发送Post请求，支持 Https
     * @param url 请求地址
     * @param params 参数: code=abc123&token=abc
     * @return  返回响应字符串
     */
    public static File postForFile(String url, String params) throws IOException {
        assertRestTemplate();

        HttpHeaders headers = new HttpHeaders();
        HttpEntity<String> request = new HttpEntity<>(params, headers);

        ResponseEntity<Resource> responseEntity = restTemplate.postForEntity(url, request, Resource.class);
        InputStream inputStream = responseEntity.getBody().getInputStream();
        BufferedInputStream bis = new BufferedInputStream(inputStream);
        File tempFile = File.createTempFile(IdGen.uuid(), ".jpeg");
        OutputStream os = new FileOutputStream(tempFile);
        int len;
        byte[] arr = new byte[1024];
        while ((len = bis.read(arr)) != -1) {
            os.write(arr, 0, len);
            os.flush();
        }
        os.close();
        bis.close();

        inputStream.close();

        return tempFile;
    }

    /**
     * 使用JSON参数发送Post请求，支持微信小程序的 Https
     * @param url 请求地址
     * @param params 参数为 JSON 格式的参数：  { "key" : 520, "key1" : 1314 }
     * @return  返回响应字符串
     */
    public static String postJsonForString(String url, String params) {
        assertRestTemplate();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<String> request = new HttpEntity<>(params, headers);

        String response = restTemplate.postForObject(url, request, String.class);
        logger.debug("response = {}", response);

        return response;
    }

    /**
     * 使用JSON参数发送Post请求，支持微信小程序的 Https
     * @param url 请求地址
     * @param params 参数为 JSON 格式的参数：  { "key" : 520, "key1" : 1314 }
     * @return  返回响应字符串
     */
    public static File postJsonForFile(String url, String params) throws IOException {
        assertRestTemplate();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON_UTF8);

        HttpEntity<String> request = new HttpEntity<>(params, headers);

        ResponseEntity<Resource> responseEntity = restTemplate.postForEntity(url, request, Resource.class);
        InputStream inputStream = responseEntity.getBody().getInputStream();
        BufferedInputStream bis = new BufferedInputStream(inputStream);
        File tempFile = File.createTempFile(IdGen.uuid(), ".jpeg");
        OutputStream os = new FileOutputStream(tempFile);
        int len;
        byte[] arr = new byte[1024];
        while ((len = bis.read(arr)) != -1) {
            os.write(arr, 0, len);
            os.flush();
        }
        os.close();
        bis.close();

        inputStream.close();

        return tempFile;
    }

    /**
     * 检查 restTemplate 不为空
     */
    private static void assertRestTemplate() {
        Assert.notNull(restTemplate, "请开启Spring注解扫描功能后重试...");
    }
}

