package cn.net.wanmo.plugin.spring.util;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

@Component
public class SpringUtil implements BeanFactoryPostProcessor, ApplicationContextAware, DisposableBean {
    /** Spring应用上下文环境 */
    private static ApplicationContext applicationContext;
    /** Spring应用上下文环境 */
    private static ConfigurableListableBeanFactory beanFactory;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        if(SpringUtil.applicationContext == null) {
            SpringUtil.applicationContext = applicationContext;
        }

        System.out.println("---------------------------------------------------------------------");
        System.out.println("========ApplicationContext 配置成功,在普通类可以通过调用 SpringUtils.getAppContext() 获取对象========");
        System.out.println("---------------------------------------------------------------------");
    }


    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        if (SpringUtil.beanFactory == null) {
            SpringUtil.beanFactory = beanFactory;
        }

        System.out.println("---------------------------------------------------------------------");
        System.out.println("========ConfigurableBeanFactory 配置成功, 在普通类可以通过调用 SpringUtils.getBeanFactory() 获取对象========");
        System.out.println("---------------------------------------------------------------------");
    }

    /** 获取 ApplicationContext */
    public static ApplicationContext getAppContext() {
        assertContextInjected();
        return applicationContext;
    }
    /** 获取 BeanFactory */
    public static ConfigurableListableBeanFactory getBeanFactory() {
        assertBeanFactoryInjected();
        return beanFactory;
    }

    /**
     * 从静态变量applicationContext中取得Bean, 自动转型为所赋值对象的类型.
     */
    public static Object getBean(String name) {
        assertContextInjected();
        return getAppContext().getBean(name);
    }

    /**
     * 从静态变量applicationContext中取得Bean, 自动转型为所赋值对象的类型.
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> T getBean(Class<T> clazz) {
        assertContextInjected();
        return getAppContext().getBean(clazz);
    }

    /**
     * 通过name,以及Clazz返回指定的Bean
     * @param name
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> T getBean(String name, Class<T> clazz) {
        assertContextInjected();
        return getAppContext().getBean(name, clazz);
    }


    /**
     * 从静态变量 BeanFactory 中取得Bean, 自动转型为所赋值对象的类型.
     * @param name
     * @return
     */
    public static Object getBeanFromFactory(String name) {
        assertBeanFactoryInjected();
        return getBeanFactory().getBean(name);
    }

    /**
     * 从静态变量 BeanFactory 中取得Bean, 自动转型为所赋值对象的类型.
     * @param clazz
     * @param <T>
     * @return
     */
    public static <T> T getBeanFromFactory(Class<T> clazz) {
        assertBeanFactoryInjected();
        return getBeanFactory().getBean(clazz);
    }

    /**
     * 检查ApplicationContext不为空.
     */
    private static void assertContextInjected() {
        if (applicationContext == null) {
            throw new RuntimeException("applicationContext 属性未注入, 请在 applicationContext.xml 中定义 SpringUtil.");
        }
    }

    /**
     * 检查 BeanFactory 不为空.
     */
    private static void assertBeanFactoryInjected() {
        if (beanFactory == null) {
            throw new RuntimeException("beanFactory 属性未注入, 请在 applicationContext.xml 中定义 SpringUtil.");
        }
    }

    /**
     * 实现DisposableBean接口, 在Context关闭时清理静态变量.
     */
    @Override
    public void destroy() throws Exception {
        applicationContext = null;
        beanFactory = null;
    }

    /**
     * 获取当前请求对象
     *
     * @return
     */
    public static HttpServletRequest getRequest() {
        try {
            return ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        } catch (Exception e) {
            return null;
        }
    }
}
