package cn.net.wanmo.common.weixin.work.inner.server_api.storage;

import cn.net.wanmo.common.pojo.InterfaceResult;
import cn.net.wanmo.common.util.DateUtil;
import cn.net.wanmo.common.util.Threads;
import cn.net.wanmo.common.weixin.work.inner.pojo.Agent;
import cn.net.wanmo.common.weixin.work.inner.pojo.token.AccessToken;
import cn.net.wanmo.common.weixin.work.inner.pojo.token.TicketToken;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.token.AccessTokenReq;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.token.AccessTokenRes;
import cn.net.wanmo.common.weixin.work.inner.pojo.Corp;
import cn.net.wanmo.common.weixin.work.inner.server_api.util.WorkInnerUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

public class AccessTokenForAgent {
    private static Logger logger = LoggerFactory.getLogger(AccessTokenForAgent.class);


    // ------------------------------- 应用 配置 start -------------------------------------------
    /**
     * 用来存储应用对应的 访问令牌 <br/>
     * key: 企业ID 和 应用ID 的组合，内置方法生成： getAgentConfigsKey(corpId, agentId) <br/>
     * value: 应用相关配置信息  <br/>
     */
    private static Map<String, Agent> agentMap = new HashMap<>();

    /**
     * 生成 key
     *
     * @param corpId  企业 ID
     * @param agentId 应用ID
     * @return key
     */
    private static String getMapKey(String corpId, String agentId) {
        return corpId + ":" + agentId;
    }

    /**
     * 设置 应用相关的 AgentConfig
     *
     * @param corpId  企业 ID
     * @param agentId 应用ID
     * @param agent   应用信息
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken> void putAgent(String corpId, String agentId, Agent<AgentToken, JsApiTicket, JsApiConfigTicket> agent) {
        agent.getAgentToken().storeToken(); // 执行自定义保存
        agentMap.put(getMapKey(corpId, agentId), agent);
    }

    /**
     * 获取 应用相关的 AgentConfig
     *
     * @param corpId  企业 ID
     * @param agentId 应用ID
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken> Agent<AgentToken, JsApiTicket, JsApiConfigTicket> getAgent(String corpId, String agentId) {
        return agentMap.getOrDefault(getMapKey(corpId, agentId), new Agent<AgentToken, JsApiTicket, JsApiConfigTicket>());
    }
    // ------------------------------- 应用 配置 end -------------------------------------------


    // ------------------------------- 令牌初始化 start -------------------------------------------
    /**
     * 获取 应用相关的 accessToken，不存在则返回空 <br/>
     * 如果令牌有效时长小于 10秒 则重新获取
     * @param corpId 企业 ID
     * @param agentId 应用ID
     * @return 应用 accessToken
     */
//    @Deprecated
//    public static AccessToken getAgentToken(String corpId, String agentId) {
//        AccessToken accessToken = new AccessToken();
//
//        Agent agentConfig = getAgentConfig(corpId, agentId);
//        if (agentConfig == null || StringUtil.isBlank(agentConfig.getAgentId())) { // 判断 AgentConfig  是否为空
//            logger.debug("企业ID: {}, 应用ID: {}, 不存在", corpId, agentId);
//            return accessToken;
//        }
//
//        accessToken = agentConfig.getToken();
//        if (accessToken == null || StringUtil.isBlank(accessToken.getToken())) { // 判断访问令牌是否为空
//            logger.debug("企业ID: {}, 应用ID: {}, 无访问令牌", corpId, agentId);
//            return accessToken;
//        }
//
//        // 有效期大于10秒，直接返回访问令牌
//        if ((accessToken.getExpiresIn() * 1000) - (DateUtil.nowLong() - accessToken.getResTime()) > 10 * 1000) {
//            logger.debug("企业ID: {}, 应用ID: {}, 访问令牌有效：{}", corpId, agentId, accessToken.getToken());
//            return accessToken;
//        }
//
//        // 重新获取令牌
//        {
//            InterfaceResult<AccessTokenRes> result = initAgentToken(corpId, agentId, agentConfig.getSecret());
//            if (result.isSuccess()) {
//                AccessTokenRes data = result.getData();
//                accessToken = new AccessToken(data.getToken(), data.getExpiresIn(), data.getResTime());
//                logger.debug("企业ID: {}, 应用ID: {}, 重新获取访问令牌：{}", corpId, agentId, accessToken.getToken());
//            }
//        }
//
//        return accessToken;
//    }

    /**
     * 获取应用令牌，并且保存令牌
     *
     * @param corp  企业 ID
     * @param agent 应用ID
     * @return 接口实际响应的结果
     */
    public static InterfaceResult<AccessTokenRes> initTokenForAgent(Corp corp, Agent agent) {
        String corpId = corp.getCorpId();
        String agentId = agent.getAgentId();
        String agentSecret = agent.getSecret();

        return initTokenForAgent(corpId, agentId, agentSecret, null);
    }

    /**
     * 获取应用令牌，并且保存令牌
     *
     * @param corp       企业 ID
     * @param agent      应用ID
     * @param agentToken 扩展的自定义 token 对象，重写 storeToken() 方法，支持自定义存储令牌
     * @return 接口实际响应的结果
     */
    public static <AgentToken extends AccessToken> InterfaceResult<AccessTokenRes> initTokenForAgent(Corp corp, Agent agent, AgentToken agentToken) {
        String corpId = corp.getCorpId();
        String agentId = agent.getAgentId();
        String agentSecret = agent.getSecret();

        return initTokenForAgent(corpId, agentId, agentSecret, agentToken);
    }

    /**
     * 获取应用令牌，并且保存令牌
     *
     * @param corpId      企业 ID
     * @param agentId     应用ID
     * @param agentSecret 应用密钥
     * @return 接口实际响应的结果
     */
    public static InterfaceResult<AccessTokenRes> initTokenForAgent(String corpId, String agentId, String agentSecret) {
        return initTokenForAgent(corpId, agentId, agentSecret, null);
    }

    /**
     * 获取应用令牌，并且保存令牌
     *
     * @param corpId      企业 ID
     * @param agentId     应用ID
     * @param agentSecret 应用密钥
     * @param agentToken  扩展的自定义 token 对象，重写 storeToken() 方法，支持自定义存储令牌
     * @return 接口实际响应的结果
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken> InterfaceResult<AccessTokenRes> initTokenForAgent(String corpId, String agentId, String agentSecret, AgentToken agentToken) {
        AccessTokenReq req = AccessTokenReq.build(corpId, agentSecret);
        InterfaceResult<AccessTokenRes> result = WorkInnerUtil.getAccessToken(req);

        AccessToken accessToken = new AccessToken();
        accessToken.setCorpId(corpId);
        accessToken.setAgentId(agentId);
        accessToken.setTokenType(AccessToken.TokenType.agent);
        accessToken.setCode(result.getCode());
        accessToken.setMsg(result.getMsg());
        accessToken.setResTime(DateUtil.nowLong());

        if (result.isSuccess()) {
            AccessTokenRes data = result.getData();
            accessToken.setValue(data.getToken());
            accessToken.setExpiresIn(data.getExpiresIn());
            accessToken.setResTime(data.getResTime());
        }

        if (Objects.isNull(agentToken)) {
            Agent agent = getAgent(corpId, agentId);
            agent.setAgentToken(accessToken);
            putAgent(corpId, agentId, agent);

            logger.debug("企业微信ID: {}, 应用ID: {}, 执行默认存储应用令牌完成 ...", corpId, agentId);
        } else {
            { // 赋值操作
                agentToken.setCorpId(corpId);
                agentToken.setAgentId(agentId);
                agentToken.setTokenType(accessToken.getTokenType());
                agentToken.setCode(accessToken.getCode());
                agentToken.setMsg(accessToken.getMsg());
                agentToken.setValue(accessToken.getValue());
                agentToken.setExpiresIn(accessToken.getExpiresIn());
                agentToken.setResTime(accessToken.getResTime());
            }

            Agent<AgentToken, JsApiTicket, JsApiConfigTicket> agent = getAgent(corpId, agentId);
            agent.setAgentToken(agentToken);
            putAgent(corpId, agentId, agent);

            logger.debug("企业微信ID: {}, 应用ID: {}, 执行自定义存储应用令牌完成 ...", corpId, agentId);
        }

        return result;
    }
    // ------------------------------- 令牌初始化 end -------------------------------------------

    /**
     * 自动定时获取 应用令牌
     *
     * @param corp  企业
     * @param agent 应用
     */
    public static void runThreadInitTokenForAgent(Corp corp, Agent agent) {
        String corpId = corp.getCorpId();
        String agentId = agent.getAgentId();
        String agentSecret = agent.getSecret();

        runThreadInitTokenForAgent(corpId, agentId, agentSecret, null, null, null);
    }

    /**
     * 自动定时获取 应用令牌
     *
     * @param corp       企业
     * @param agent      应用
     * @param agentToken 扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <AgentToken extends AccessToken> void runThreadInitTokenForAgent(Corp corp, Agent agent, AgentToken agentToken) {
        String corpId = corp.getCorpId();
        String agentId = agent.getAgentId();
        String agentSecret = agent.getSecret();

        runThreadInitTokenForAgent(corpId, agentId, agentSecret, null, null, agentToken);
    }

    /**
     * 自动定时获取 应用令牌
     *
     * @param corp          企业
     * @param agent         应用
     * @param residueSecond 成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（默认 60秒）
     */
    public static void runThreadInitTokenForAgent(Corp corp, Agent agent, Integer residueSecond, Integer waitSecond) {
        String corpId = corp.getCorpId();
        String agentId = agent.getAgentId();
        String agentSecret = agent.getSecret();

        runThreadInitTokenForAgent(corpId, agentId, agentSecret, residueSecond, waitSecond, null);
    }

    /**
     * 自动定时获取 应用令牌
     *
     * @param corp          企业
     * @param agent         应用
     * @param agentToken    扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     * @param residueSecond 成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（默认 60秒）
     */
    public static <AgentToken extends AccessToken> void runThreadInitTokenForAgent(Corp corp, Agent agent, Integer residueSecond, Integer waitSecond, AgentToken agentToken) {
        String corpId = corp.getCorpId();
        String agentId = agent.getAgentId();
        String agentSecret = agent.getSecret();

        runThreadInitTokenForAgent(corpId, agentId, agentSecret, residueSecond, waitSecond, agentToken);
    }

    /**
     * 自动定时获取 应用令牌
     *
     * @param corpId      企业 ID
     * @param agentId     应用ID
     * @param agentSecret 应用密钥
     */
    public static void runThreadInitTokenForAgent(String corpId, String agentId, String agentSecret) {
        runThreadInitTokenForAgent(corpId, agentId, agentSecret, null, null, null);
    }

    /**
     * 自动定时获取 应用令牌
     *
     * @param corpId      企业 ID
     * @param agentId     应用ID
     * @param agentSecret 应用密钥
     * @param agentToken  扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <AgentToken extends AccessToken> void runThreadInitTokenForAgent(String corpId, String agentId, String agentSecret, AgentToken agentToken) {
        runThreadInitTokenForAgent(corpId, agentId, agentSecret, null, null, agentToken);
    }


    /**
     * 自动定时获取 应用令牌
     *
     * @param corpId        企业 ID
     * @param agentId       应用ID
     * @param agentSecret   应用密钥
     * @param residueSecond 成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（默认 60秒）
     */
    public static void runThreadInitTokenForAgent(String corpId, String agentId, String agentSecret, Integer residueSecond, Integer waitSecond) {
        runThreadInitTokenForAgent(corpId, agentId, agentSecret, residueSecond, waitSecond, null);
    }

    /**
     * 自动定时获取 应用令牌
     *
     * @param corpId        企业 ID
     * @param agentId       应用ID
     * @param agentSecret   应用密钥
     * @param residueSecond 成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（默认 60秒）
     * @param agentToken    扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <AgentToken extends AccessToken> void runThreadInitTokenForAgent(String corpId, String agentId, String agentSecret, Integer residueSecond, Integer waitSecond, AgentToken agentToken) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                while (true) {
                    try {
                        InterfaceResult<AccessTokenRes> result = initTokenForAgent(corpId, agentId, agentSecret, agentToken);
                        if (result.isSuccess()) { // 令牌获取成功
                            String token = result.getData().getToken();
                            Integer expiresIn = result.getData().getExpiresIn();

                            logger.debug("企业微信ID: {}, 应用ID: {}, 定时任务获取 应用令牌 成功，expiresIn = {}， token = {}", corpId, agentId, expiresIn, token);
                            Threads.sleep((expiresIn - (residueSecond == null ? 30 : residueSecond)) * 1000); // 休眠到 有效期剩余 30秒
                        } else { // 令牌获取失败
                            logger.error("企业微信ID: {}, 应用ID: {}, 定时任务获取 应用令牌 失败, errCode:{} errMsg:{}", corpId, agentId, result.getCode(), result.getMsg());
                            Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                        }
                    } catch (Exception e) {
                        logger.error("企业微信ID: " + corpId + ", 应用ID: " + agentId + ", 定时任务获取 应用令牌 异常", e);
                        Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                    }
                }
            }
        }).start();
    }

    // ------------------------------- 应用 令牌 end -------------------------------------------
}
