package cn.net.wanmo.plugin.wechat.officialaccount.storage;

import cn.net.wanmo.common.result.InterfaceResult;
import cn.net.wanmo.common.util.BooleanUtil;
import cn.net.wanmo.common.util.Threads;
import cn.net.wanmo.plugin.wechat.officialaccount.util.accesstoken.AccessToken;
import cn.net.wanmo.plugin.wechat.officialaccount.util.accesstoken.AccessTokenRes;
import cn.net.wanmo.plugin.wechat.officialaccount.util.accesstoken.StableAccessTokenRes;
import cn.net.wanmo.plugin.wechat.officialaccount.util.accesstoken.TokenUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 访问令牌生成和存储
 */
public class AccessTokenStorage {
    private static Logger logger = LoggerFactory.getLogger(AccessTokenStorage.class);

    /**
     * 访问令牌 存储
     */
    private static Map<String, AccessToken> tokenMap = new HashMap<>();

    /**
     * 设置访问令牌
     *
     * @param appId         应用ID
     * @param myAccessToken 访问令牌
     */
    public static <MyAccessToken extends AccessToken> void putToken(String appId, MyAccessToken myAccessToken) {
        myAccessToken.storeToken(); // 执行自定义保存令牌
        tokenMap.put(appId, myAccessToken);
    }

    /**
     * 获取访问令牌
     *
     * @param appId 应用ID
     * @return 访问令牌
     */
    public static AccessToken getToken(String appId) {
        return tokenMap.getOrDefault(appId, new AccessToken());
    }


    /**
     * 获取通讯录令牌，并且保存令牌
     *
     * @param appId         公众号ID
     * @param appSecret     公众号密钥
     * @param myAccessToken 扩展的自定义 token 对象，重写 storeToken() 方法，支持自定义存储令牌
     * @return 接口实际响应的结果
     */
    public static <MyAccessToken extends AccessToken> InterfaceResult<AccessTokenRes> initToken(String appId, String appSecret, MyAccessToken myAccessToken) {
        InterfaceResult<AccessTokenRes> result = TokenUtil.getAccessToken(appId, appSecret);

        if (BooleanUtil.isFalse(result.isSuccess())) {
            return result;
        }

        AccessToken accessToken = result.getData().getAccessToken();

        if (Objects.isNull(myAccessToken)) {
            putToken(appId, accessToken);
        } else {
            myAccessToken.setToken(accessToken.getToken());
            myAccessToken.setExpiresIn(accessToken.getExpiresIn());

            putToken(appId, myAccessToken);
        }

        // 内存中记录日志
        AccessTokenLog.put(appId, accessToken);
        return result;
    }


    /**
     * 自动定时获取 通讯录令牌
     *
     * @param appId         公众号ID
     * @param appSecret     公众号密钥
     * @param residueSecond 成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（默认 60秒）
     * @param myAccessToken 扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <MyAccessToken extends AccessToken> void runThreadInitToken(final String appId, final String appSecret, Integer residueSecond, Integer waitSecond, MyAccessToken myAccessToken) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                while (true) {
                    try {
                        InterfaceResult<AccessTokenRes> result = initToken(appId, appSecret, myAccessToken);

                        if (result.isSuccess()) { // 令牌获取成功
                            AccessToken accessToken = result.getData().getAccessToken();
                            String token = accessToken.getToken();
                            int expiresIn = accessToken.getExpiresIn();

                            logger.debug("微信公众号ID: {}, 定时任务获取 访问令牌 成功，expiresIn = {}， token = {}", appId, expiresIn, token);
                            Threads.sleep((expiresIn - (residueSecond == null ? 30 : residueSecond)) * 1000); // 休眠到 有效期剩余 30秒
                        } else { // 令牌获取失败
                            logger.error("微信公众号ID: {}, 定时任务获取 访问令牌 失败, errCode:{} errMsg:{}", appId, result.getCode(), result.getMsg());
                            Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                        }
                    } catch (Exception e) {
                        logger.error("微信公众号ID: " + appId + ", 定时任务获取 访问令牌 异常", e);
                        Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                    }
                }
            }
        }).start();
    }


    /**
     * 获取通讯录令牌，并且保存令牌
     *
     * @param appId         公众号ID
     * @param appSecret     公众号密钥
     * @param myAccessToken 扩展的自定义 token 对象，重写 storeToken() 方法，支持自定义存储令牌
     * @return 接口实际响应的结果
     */
    public static <MyAccessToken extends AccessToken> InterfaceResult<StableAccessTokenRes> initStableAccessToken(String appId, String appSecret, MyAccessToken myAccessToken) {
        InterfaceResult<StableAccessTokenRes> result = TokenUtil.getStableAccessToken(appId, appSecret);

        if (BooleanUtil.isFalse(result.isSuccess())) {
            return result;
        }

        AccessToken accessToken = result.getData().getAccessToken();

        if (Objects.isNull(myAccessToken)) {
            putToken(appId, accessToken);
        } else {
            myAccessToken.setToken(accessToken.getToken());
            myAccessToken.setExpiresIn(accessToken.getExpiresIn());

            putToken(appId, myAccessToken);
        }

        // 内存中记录日志
        AccessTokenLog.put(appId, accessToken);
        return result;
    }


    /**
     * 自动定时获取 通讯录令牌
     *
     * @param appId         公众号ID
     * @param appSecret     公众号密钥
     * @param residueSecond 成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（默认 60秒）
     * @param myAccessToken 扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <MyAccessToken extends AccessToken> void runThreadInitStableAccessToken(final String appId, final String appSecret, Integer residueSecond, Integer waitSecond, MyAccessToken myAccessToken) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                while (true) {
                    try {
                        InterfaceResult<StableAccessTokenRes> result = initStableAccessToken(appId, appSecret, myAccessToken);

                        if (result.isSuccess()) { // 令牌获取成功
                            AccessToken accessToken = result.getData().getAccessToken();
                            String token = accessToken.getToken();
                            int expiresIn = accessToken.getExpiresIn();

                            logger.debug("微信公众号ID: {}, 定时任务获取 访问令牌 成功，expiresIn = {}， token = {}", appId, expiresIn, token);
                            Threads.sleep((expiresIn - (residueSecond == null ? 30 : residueSecond)) * 1000); // 休眠到 有效期剩余 30秒
                        } else { // 令牌获取失败
                            logger.error("微信公众号ID: {}, 定时任务获取 访问令牌 失败, errCode:{} errMsg:{}", appId, result.getCode(), result.getMsg());
                            Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                        }
                    } catch (Exception e) {
                        logger.error("微信公众号ID: " + appId + ", 定时任务获取 访问令牌 异常", e);
                        Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                    }
                }
            }
        }).start();
    }
}
