package cn.net.wanmo.plugin.wechat.officialaccount.storage;

import cn.net.wanmo.common.result.InterfaceResult;
import cn.net.wanmo.common.util.BooleanUtil;
import cn.net.wanmo.common.util.Threads;
import cn.net.wanmo.plugin.wechat.officialaccount.util.jsapiticket.JsapiTicket;
import cn.net.wanmo.plugin.wechat.officialaccount.util.jsapiticket.JsapiTicketRes;
import cn.net.wanmo.plugin.wechat.officialaccount.util.jsapiticket.TicketUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * JsapiTicket 令牌生成和存储
 */
public class JsapiTicketStorage {
    private static Logger logger = LoggerFactory.getLogger(JsapiTicketStorage.class);

    /**
     * 访问令牌 存储
     */
    private static Map<String, JsapiTicket> ticketMap = new HashMap<>();

    /**
     * 设置访问令牌
     *
     * @param appId         应用ID
     * @param myJsapiTicket 访问令牌
     */
    public static <MyJsapiTicket extends JsapiTicket> void putTicket(String appId, MyJsapiTicket myJsapiTicket) {
        myJsapiTicket.storeTicket(); // 执行自定义保存令牌
        ticketMap.put(appId, myJsapiTicket);
    }

    /**
     * 获取访问令牌
     *
     * @param appId 应用ID
     * @return 访问令牌
     */
    public static JsapiTicket getTicket(String appId) {
        return ticketMap.getOrDefault(appId, new JsapiTicket());
    }


    /**
     * 获取通讯录令牌，并且保存令牌
     *
     * @param appId         公众号ID
     * @param accessToken   访问令牌
     * @param myJsapiTicket 扩展的自定义 token 对象，重写 storeTicket() 方法，支持自定义存储令牌
     * @return 接口实际响应的结果
     */
    public static <MyJsapiTicket extends JsapiTicket> InterfaceResult<JsapiTicketRes> initTicket(String appId, String accessToken, MyJsapiTicket myJsapiTicket) {
        InterfaceResult<JsapiTicketRes> result = TicketUtil.getJsapiTicket("", accessToken);

        if (BooleanUtil.isFalse(result.isSuccess())) {
            return result;
        }

        JsapiTicket jsapiTicket = result.getData().getJsapiTicket();

        if (Objects.isNull(myJsapiTicket)) {
            putTicket(appId, jsapiTicket);
        } else {
            myJsapiTicket.setTicket(jsapiTicket.getTicket());
            myJsapiTicket.setExpiresIn(jsapiTicket.getExpiresIn());

            putTicket(appId, myJsapiTicket);
        }

        // 内存中记录日志
        JsapiTicketLog.put(appId, jsapiTicket);
        return result;
    }


    /**
     * 自动定时获取 通讯录令牌
     *
     * @param appId         公众号ID
     * @param accessToken   访问令牌
     * @param residueSecond 成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（默认 60秒）
     * @param myJsapiTicket 扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <MyJsapiTicket extends JsapiTicket> void runThreadInitToken(final String appId, final String accessToken, Integer residueSecond, Integer waitSecond, MyJsapiTicket myJsapiTicket) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                while (true) {
                    try {
                        InterfaceResult<JsapiTicketRes> result = initTicket(appId, accessToken, myJsapiTicket);

                        if (result.isSuccess()) { // 令牌获取成功
                            JsapiTicket jsapiTicket = result.getData().getJsapiTicket();
                            String ticket = jsapiTicket.getTicket();
                            int expiresIn = jsapiTicket.getExpiresIn();

                            logger.debug("微信公众号ID: {}, 定时任务获取 jsapiTicket 成功，expiresIn = {}， ticket = {}", appId, expiresIn, ticket);
                            Threads.sleep((expiresIn - (residueSecond == null ? 30 : residueSecond)) * 1000); // 休眠到 有效期剩余 30秒
                        } else { // 令牌获取失败
                            logger.error("微信公众号ID: {}, 定时任务获取 jsapiTicket 失败, errCode:{} errMsg:{}", appId, result.getCode(), result.getMsg());
                            Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                        }
                    } catch (Exception e) {
                        logger.error("微信公众号ID: " + appId + ", 定时任务获取 访问令牌 异常", e);
                        Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                    }
                }
            }
        }).start();
    }
}
