package cn.net.wanmo.plugin.wechat.officialaccount.core.service;


import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import cn.net.wanmo.common.util.StringUtil;
import cn.net.wanmo.plugin.wechat.officialaccount.util.menu.MenuEventType;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.event.req.ReqEventType;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.event.req.account.AccountSubscribeMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.event.req.account.AccountUnsubscribeMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.event.req.location.ReportLocationMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.event.req.menu.*;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.event.req.scan.ScanSubscribeNoMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.event.req.scan.ScanSubscribeYesMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.ReqMessageType;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.ReqMessageUtil;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.base.PlainReqMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.image.PlainReqImageMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.link.PlainReqLinkMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.location.PlainReqLocationMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.shortvideo.PlainReqShortVideoMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.text.PlainReqTextMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.video.PlainReqVideoMessage;
import cn.net.wanmo.plugin.wechat.officialaccount.util.message.plain.req.voice.PlainReqVoiceMessage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 实现该类，根据消息类型，处理请求的消息
 */
public abstract class CoreService {
    protected Logger logger = LoggerFactory.getLogger(getClass());

    /**
     * 微信公众号服务端 验证三方URL配置 时用作生成签名，需要使用该 token
     */
    protected String token;

    public CoreService() {
        init();
    }


    public String getToken() {
        return token;
    }

    public void setToken(String token) {
        this.token = token;
    }

    public void init() {
    }

    public String processRequest(HttpServletRequest request) {
        String resXml = "success";
        try {
            logger.debug("==================== 接收到腾讯公众号服务器的请求，处理开始 ==================== ");

            Map<String, String> requestMap = ReqMessageUtil.parseXml2Map(request);
            ReqMessageType msgType = PlainReqMessage.getMsgType(requestMap);
            logger.debug("微信公众号 接收普通消息类型：{}", msgType);

            switch (msgType) {
                case text: { // 文本消息
                    logger.debug("微信公众号 处理普通文本消息类型");
                    PlainReqTextMessage message = processTextMessage(requestMap);
                    resXml = message.getPlainResXml();
                    break;
                }
                case image: { // 图片消息
                    logger.debug("微信公众号 处理普通图片消息类型");
                    PlainReqImageMessage message = processImageMessage(requestMap);
                    resXml = message.getPlainResXml();
                    break;
                }
                case voice: { // 音频消息
                    logger.debug("微信公众号 处理普通音频消息类型");
                    PlainReqVoiceMessage message = processVoiceMessage(requestMap);
                    resXml = message.getPlainResXml();
                    break;
                }
                case video: {
                    logger.debug("微信公众号 处理普通视频消息类型");
                    PlainReqVideoMessage message = processVideoMessage(requestMap);
                    resXml = message.getPlainResXml();
                    break;
                }
                case shortvideo: {
                    logger.debug("微信公众号 处理普通短视频消息类型");
                    PlainReqShortVideoMessage message = processShortvideoMessage(requestMap);
                    resXml = message.getPlainResXml();
                    break;
                }
                case location: {
                    logger.debug("微信公众号 处理普通位置消息类型");
                    PlainReqLocationMessage message = processLocationMessage(requestMap);
                    resXml = message.getPlainResXml();
                    break;
                }
                case link: {
                    logger.debug("微信公众号 处理普通链接消息类型");
                    PlainReqLinkMessage message = processLinkMessage(requestMap);
                    resXml = message.getPlainResXml();
                    break;
                }
                case event: {
                    String eventType = requestMap.get("Event");
                    logger.debug("微信公众号 处理普通事件消息类型：{}", eventType);

                    resXml = plainEvent(requestMap, eventType);
                    break;
                }
                default: {
                    logger.warn("微信公众号 未知的消息类型：{}", msgType);
                }
            }
        } catch (Exception e) {
            logger.error("根据消息类型解析响应异常", e);
        }
        return resXml;
    }

    /**
     * 处理普通事件消息类型
     */
    private String plainEvent(Map<String, String> requestMap, String eventType) {
        String resXml = "success";

        ReqEventType reqEventType = ReqEventType.convertOf(eventType);
        switch (reqEventType) {
            case subscribe: {
                String ticket = requestMap.get("Ticket");
                if (StringUtil.isBlank(ticket)) { // 触发关注事件
                    AccountSubscribeMessage message = processAccountSubscribeMessage(requestMap);
                    resXml = message.getPlainResXml();
                } else { // 未关注 扫码事件
                    ScanSubscribeNoMessage message = processScanSubscribeNoMessage(requestMap);
                    resXml = message.getPlainResXml();
                }
                break;
            }
            case unsubscribe: {
                AccountUnsubscribeMessage message = processAccountUnsubscribeMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case SCAN: { // 已关注 扫码事件
                ScanSubscribeYesMessage message = processScanSubscribeYesMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case LOCATION: {
                ReportLocationMessage message = processReportLocationMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            default: {
                resXml = menuEvent(requestMap, eventType);
            }
        }

        return resXml;
    }

    /**
     * 处理菜单事件消息类型
     */
    private String menuEvent(Map<String, String> requestMap, String eventType) {
        String resXml = "success";

        MenuEventType menuEventType = MenuEventType.convertOf(eventType);
        switch (menuEventType) {
            case CLICK: {
                MenuClickMessage message = processMenuClickMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case VIEW: {
                MenuViewMessage message = processMenuViewMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case view_miniprogram: {
                MenuViewMiniprogramMessage message = processMenuViewMiniprogramMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case location_select: {
                MenuLocationSelectMessage message = processMenuLocationSelectMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case scancode_push: {
                MenuScancodePushMessage message = processMenuScancodePushMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case scancode_waitmsg: {
                MenuScancodeWaitmsgMessage message = processMenuScancodeWaitmsgMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case pic_weixin: {
                MenuPicWeixinMessage message = processMenuPicWeixinMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case pic_sysphoto: {
                MenuPicSysphotoMessage message = processMenuPicSysphotoMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            case pic_photo_or_album: {
                MenuPicPhotoOrAlbumMessage message = processMenuPicPhotoOrAlbumMessage(requestMap);
                resXml = message.getPlainResXml();
                break;
            }
            default: {
                logger.warn("微信公众号 未知的事件消息类型：{}", menuEventType);
            }
        }
        return resXml;
    }

    /**
     * 处理接收的文本消息
     */
    protected PlainReqTextMessage processTextMessage(Map<String, String> requestMap) {
        return new PlainReqTextMessage(requestMap);
    }

    /**
     * 处理接收的图片消息
     */
    protected PlainReqImageMessage processImageMessage(Map<String, String> requestMap) {
        return new PlainReqImageMessage(requestMap);
    }

    /**
     * 处理接收的语音消息
     */
    protected PlainReqVoiceMessage processVoiceMessage(Map<String, String> requestMap) {
        return new PlainReqVoiceMessage(requestMap);
    }

    /**
     * 处理接收的视频消息
     */
    protected PlainReqVideoMessage processVideoMessage(Map<String, String> requestMap) {
        return new PlainReqVideoMessage(requestMap);
    }

    /**
     * 处理接收的小视频消息
     */
    protected PlainReqShortVideoMessage processShortvideoMessage(Map<String, String> requestMap) {
        return new PlainReqShortVideoMessage(requestMap);
    }

    /**
     * 处理接收的地理位置消息
     */
    protected PlainReqLocationMessage processLocationMessage(Map<String, String> requestMap) {
        return new PlainReqLocationMessage(requestMap);
    }

    /**
     * 处理接收的连接消息
     */
    protected PlainReqLinkMessage processLinkMessage(Map<String, String> requestMap) {
        return new PlainReqLinkMessage(requestMap);
    }

    /**
     * 处理接收的订阅事件
     */
    protected AccountSubscribeMessage processAccountSubscribeMessage(Map<String, String> requestMap) {
        return new AccountSubscribeMessage(requestMap);
    }

    /**
     * 处理接收的取消订阅事件
     */
    protected AccountUnsubscribeMessage processAccountUnsubscribeMessage(Map<String, String> requestMap) {
        return new AccountUnsubscribeMessage(requestMap);
    }

    /**
     * 处理接收的上报地理位置事件
     */
    protected ReportLocationMessage processReportLocationMessage(Map<String, String> requestMap) {
        return new ReportLocationMessage(requestMap);
    }

    /**
     * 处理扫描带参数二维码事件消息
     */
    protected ScanSubscribeNoMessage processScanSubscribeNoMessage(Map<String, String> requestMap) {
        return new ScanSubscribeNoMessage(requestMap);
    }

    /**
     * 处理扫描带参数二维码事件消息
     */
    protected ScanSubscribeYesMessage processScanSubscribeYesMessage(Map<String, String> requestMap) {
        return new ScanSubscribeYesMessage(requestMap);
    }

    /**
     * 处理点击菜单拉取消息时的事件推送事件消息
     */
    protected MenuClickMessage processMenuClickMessage(Map<String, String> requestMap) {
        return new MenuClickMessage(requestMap);
    }

    /**
     * 处理点击菜单跳转链接时的事件消息
     */
    protected MenuViewMessage processMenuViewMessage(Map<String, String> requestMap) {
        return new MenuViewMessage(requestMap);
    }

    /**
     * 处理点击菜单跳转小程序的事件推送
     */
    protected MenuViewMiniprogramMessage processMenuViewMiniprogramMessage(Map<String, String> requestMap) {
        return new MenuViewMiniprogramMessage(requestMap);
    }

    /**
     * 处理点击菜单弹出地理位置选择器的事件推送
     */
    protected MenuLocationSelectMessage processMenuLocationSelectMessage(Map<String, String> requestMap) {
        return new MenuLocationSelectMessage(requestMap);
    }

    /**
     * 处理点击菜单弹出拍照或者相册发图的事件推送
     */
    protected MenuPicPhotoOrAlbumMessage processMenuPicPhotoOrAlbumMessage(Map<String, String> requestMap) {
        return new MenuPicPhotoOrAlbumMessage(requestMap);
    }

    /**
     * 处理点击菜单弹出系统拍照发图的事件推送
     */
    protected MenuPicSysphotoMessage processMenuPicSysphotoMessage(Map<String, String> requestMap) {
        return new MenuPicSysphotoMessage(requestMap);
    }

    /**
     * 处理点击菜单弹出微信相册发图器的事件推送
     */
    protected MenuPicWeixinMessage processMenuPicWeixinMessage(Map<String, String> requestMap) {
        return new MenuPicWeixinMessage(requestMap);
    }

    /**
     * 处理点击菜单扫码推事件的事件推送
     */
    protected MenuScancodePushMessage processMenuScancodePushMessage(Map<String, String> requestMap) {
        return new MenuScancodePushMessage(requestMap);
    }

    /**
     * 处理点击菜单扫码推事件且弹出“消息接收中”提示框的事件推送
     */
    protected MenuScancodeWaitmsgMessage processMenuScancodeWaitmsgMessage(Map<String, String> requestMap) {
        return new MenuScancodeWaitmsgMessage(requestMap);
    }

    protected void createMenu() {
    }
}

