package cn.net.wanmo.common.weixin.work.inner.client_api.storage;

import cn.net.wanmo.common.pojo.InterfaceResult;
import cn.net.wanmo.common.util.DateUtil;
import cn.net.wanmo.common.util.Threads;
import cn.net.wanmo.common.weixin.work.inner.pojo.log.TokenLog;
import cn.net.wanmo.common.weixin.work.inner.pojo.token.TicketToken;
import cn.net.wanmo.common.weixin.work.inner.client_api.pojo.jsapi_ticket.JsapiTicketReqForAgent;
import cn.net.wanmo.common.weixin.work.inner.client_api.pojo.jsapi_ticket.JsapiTicketResForAgent;
import cn.net.wanmo.common.weixin.work.inner.client_api.util.jsapi_ticket.JsapiTicketUtil;
import cn.net.wanmo.common.weixin.work.inner.pojo.token.AccessToken;
import cn.net.wanmo.common.weixin.work.inner.pojo.Agent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;


/**
 * 应用票据令牌的定时获取
 */
public class TicketTokenForAgent {
    private static Logger logger = LoggerFactory.getLogger(TicketTokenForAgent.class);


    // ------------------------------- 令牌存储 start -------------------------------------------
    /**
     * 存储令牌 <br/>
     * key: 应用ID  <br/>
     * value: 令牌信息  <br/>
     */
    private static Map<String, TicketToken> ticketMap = new HashMap<>();

    /**
     * 生成 key
     *
     * @param corpId  企业 ID
     * @param agentId 应用ID
     * @return key
     */
    private static String getMapKey(String corpId, String agentId) {
        return corpId + ":" + agentId;
    }

    /**
     * 设置令牌
     *
     * @param corpId           企业 ID
     * @param agentId          应用ID
     * @param ticketTokenAgent 票据令牌
     */
    public static <TicketTokenAgent extends TicketToken> void putTicket(String corpId, String agentId, TicketTokenAgent ticketTokenAgent) {
        ticketTokenAgent.storeToken(); // 执行自定义存储
        ticketMap.put(getMapKey(corpId, agentId), ticketTokenAgent);
    }

    /**
     * 获取令牌
     *
     * @param corpId  企业 ID
     * @param agentId 应用ID
     * @return 票据令牌
     */
    public static TicketToken getTicket(String corpId, String agentId) {
        return ticketMap.getOrDefault(getMapKey(corpId, agentId), new TicketToken());
    }
    // ------------------------------- 令牌存储 end -------------------------------------------

    // ------------------------------- 令牌初始化 start -------------------------------------------

    /**
     * 获取令牌，并且保存令牌
     *
     * @param agent 应用信息，需包括企业信息和应用令牌
     * @return 接口实际响应的结果
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken> InterfaceResult<JsapiTicketResForAgent> initToken(Agent<AgentToken, JsApiTicket, JsApiConfigTicket> agent) {
        String corpId = agent.getCorpId();
        String agentId = agent.getAgentId();
        String logPrevDesc = agent.getLogPrevDesc();
        AgentToken agentToken = agent.getAgentToken();
        return initToken(corpId, agentId, logPrevDesc, agentToken, null);
    }

    /**
     * 获取令牌，并且保存令牌
     *
     * @param agent 应用信息，需包括企业信息和应用令牌
     * @return 接口实际响应的结果
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken, Ticket extends TicketToken> InterfaceResult<JsapiTicketResForAgent> initToken(Agent<AgentToken, JsApiTicket, JsApiConfigTicket> agent, Ticket ticket) {
        String corpId = agent.getCorpId();
        String agentId = agent.getAgentId();
        String logPrevDesc = agent.getLogPrevDesc();
        AgentToken agentToken = agent.getAgentToken();
        return initToken(corpId, agentId, logPrevDesc, agentToken, ticket);
    }

    /**
     * 获取令牌，并且保存令牌
     *
     * @param agentId     企业 ID
     * @param agentId     应用 ID
     * @param logPrevDesc 日志前缀描述
     * @param agentToken  应用令牌
     * @return 接口实际响应的结果
     */
    public static <AgentToken extends AccessToken> InterfaceResult<JsapiTicketResForAgent> initToken(String corpId, String agentId, String logPrevDesc, AgentToken agentToken) {
        return initToken(corpId, agentId, logPrevDesc, agentToken, null);
    }

    /**
     * 获取令牌，并且保存令牌
     *
     * @param agentId     企业 ID
     * @param agentId     应用 ID
     * @param logPrevDesc 日志前缀描述
     * @param agentToken  应用令牌
     * @param ticket      扩展的自定义 ticket 对象，重写 storeToken() 方法，支持自定义存储令牌
     * @return 接口实际响应的结果
     */
    public static <AgentToken extends AccessToken, Ticket extends TicketToken> InterfaceResult<JsapiTicketResForAgent> initToken(String corpId, String agentId, String logPrevDesc, AgentToken agentToken, Ticket ticket) {
        JsapiTicketReqForAgent req = JsapiTicketReqForAgent.build();
        String accessToken = agentToken.takeToken();
        InterfaceResult<JsapiTicketResForAgent> result = JsapiTicketUtil.getJsapiTicketForAgent(logPrevDesc, accessToken, req);

        TicketToken ticketToken = new TicketToken();
        ticketToken.setCorpId(corpId);
        ticketToken.setAgentId(agentId);
        ticketToken.setTokenType(TicketToken.TokenType.agent);
        ticketToken.setCode(result.getCode());
        ticketToken.setMsg(result.getMsg());
        ticketToken.setResTime(DateUtil.nowLong());

        if (result.isSuccess()) {
            JsapiTicketResForAgent data = result.getData();
            ticketToken.setValue(data.getTicket());
            ticketToken.setExpiresIn(data.getExpiresIn());
            ticketToken.setResTime(data.getResTime());
        }

        if (Objects.isNull(ticket)) {
            putTicket(corpId, agentId, ticketToken);
            logger.debug("企业应用ID: {}, 执行默认存储应用 jsApi 票据令牌 ...", agentId);
        } else {
            { // 赋值操作
                ticket.setCorpId(ticketToken.getCorpId());
                ticket.setAgentId(ticketToken.getAgentId());
                ticket.setTokenType(ticketToken.getTokenType());
                ticket.setCode(ticketToken.getCode());
                ticket.setMsg(ticketToken.getMsg());
                ticket.setValue(ticketToken.getValue());
                ticket.setExpiresIn(ticketToken.getExpiresIn());
                ticket.setResTime(ticketToken.getResTime());
            }
            putTicket(corpId, agentId, ticket);
            logger.debug("企业应用ID: {}, 执行自定义存储应用 jsApi 票据令牌 ...", agentId);
        }

        { // 内存中记录日志
            TokenLog.add(corpId, agentId, ticketToken.getTokenType().getLabel(), ticketToken, TokenLog.ticketTokens);
        }

        return result;
    }
    // ------------------------------- 令牌初始化 end -------------------------------------------


    // ------------------------------- 令牌线程 start -------------------------------------------

    /**
     * 自动定时获取 令牌
     *
     * @param agent 应用信息，需包括企业信息和应用令牌
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken> void runThreadInitToken(Agent<AgentToken, JsApiTicket, JsApiConfigTicket> agent) {
        String corpId = agent.getCorpId();
        String agentId = agent.getAgentId();
        String logPrevDesc = agent.getLogPrevDesc();
        AgentToken agentToken = agent.getAgentToken();

        runThreadInitToken(corpId, agentId, logPrevDesc, agentToken, null, null, null);
    }

    /**
     * 自动定时获取 令牌
     *
     * @param agent         应用信息，需包括企业信息和应用令牌
     * @param residueSecond 成功后，剩余多少时间后重新获取（秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（秒）
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken> void runThreadInitToken(Agent<AgentToken, JsApiTicket, JsApiConfigTicket> agent, Integer residueSecond, Integer waitSecond) {
        String corpId = agent.getCorpId();
        String agentId = agent.getAgentId();
        String logPrevDesc = agent.getLogPrevDesc();
        AgentToken agentToken = agent.getAgentToken();

        runThreadInitToken(corpId, agentId, logPrevDesc, agentToken, residueSecond, waitSecond, null);
    }

    /**
     * 自动定时获取 令牌
     *
     * @param agent            应用信息，需包括企业信息和应用令牌
     * @param ticketTokenAgent 扩展的自定义 ticket 对象，重写 storeToken() 方法，支持自定义存储令牌
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken, TicketTokenAgent extends TicketToken> void runThreadInitToken(Agent<AgentToken, JsApiTicket, JsApiConfigTicket> agent, TicketTokenAgent ticketTokenAgent) {
        String corpId = agent.getCorpId();
        String agentId = agent.getAgentId();
        String logPrevDesc = agent.getLogPrevDesc();
        AgentToken agentToken = agent.getAgentToken();

        runThreadInitToken(corpId, agentId, logPrevDesc, agentToken, null, null, ticketTokenAgent);
    }

    /**
     * 自动定时获取 令牌
     *
     * @param agent            应用信息，需包括企业信息和应用令牌
     * @param residueSecond    成功后，剩余多少时间后重新获取（秒）
     * @param waitSecond       失败后，等待多少时间后重新获取（秒）
     * @param ticketTokenAgent 扩展的自定义 ticket 对象，重写 storeToken() 方法，支持自定义存储令牌
     */
    public static <AgentToken extends AccessToken, JsApiTicket extends TicketToken, JsApiConfigTicket extends TicketToken, TicketTokenAgent extends TicketToken> void runThreadInitToken(Agent<AgentToken,JsApiTicket, JsApiConfigTicket> agent, Integer residueSecond, Integer waitSecond, TicketTokenAgent ticketTokenAgent) {
        String corpId = agent.getCorpId();
        String agentId = agent.getAgentId();
        String logPrevDesc = agent.getLogPrevDesc();
        AgentToken agentToken = agent.getAgentToken();

        runThreadInitToken(corpId, agentId, logPrevDesc, agentToken, residueSecond, waitSecond, ticketTokenAgent);
    }

    /**
     * 自动定时获取 令牌
     *
     * @param corpId      企业 ID
     * @param agentId     应用 ID
     * @param logPrevDesc 日志前缀描述
     * @param agentToken  应用令牌
     */
    public static <AgentToken extends AccessToken> void runThreadInitToken(String corpId, String agentId, String logPrevDesc, AgentToken agentToken) {
        runThreadInitToken(corpId, agentId, logPrevDesc, agentToken, null, null, null);
    }

    /**
     * 自动定时获取 令牌
     *
     * @param corpId        企业 ID
     * @param agentId       应用 ID
     * @param logPrevDesc   日志前缀描述
     * @param agentToken    应用令牌
     * @param residueSecond 成功后，剩余多少时间后重新获取（秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（秒）
     */
    public static <AgentToken extends AccessToken> void runThreadInitToken(String corpId, String agentId, String logPrevDesc, AgentToken agentToken, Integer residueSecond, Integer waitSecond) {
        runThreadInitToken(corpId, agentId, logPrevDesc, agentToken, residueSecond, waitSecond, null);
    }


    /**
     * 自动定时获取 令牌
     *
     * @param corpId           企业 ID
     * @param agentId          应用 ID
     * @param logPrevDesc      日志前缀描述
     * @param agentToken       应用令牌
     * @param ticketTokenAgent 扩展的自定义 ticket 对象，重写 storeToken() 方法，支持自定义存储令牌
     */
    public static <AgentToken extends AccessToken, TicketTokenAgent extends TicketToken> void runThreadInitToken(String corpId, String agentId, String logPrevDesc, AgentToken agentToken, TicketTokenAgent ticketTokenAgent) {
        runThreadInitToken(corpId, agentId, logPrevDesc, agentToken, null, null, ticketTokenAgent);
    }


    /**
     * 自动定时获取 令牌
     *
     * @param corpId           企业 ID
     * @param agentId          应用 ID
     * @param logPrevDesc      日志前缀描述
     * @param agentToken       应用令牌
     * @param residueSecond    成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond       失败后，等待多少时间后重新获取（默认 60秒）
     * @param ticketTokenAgent 扩展的自定义 ticket 对象，重写 storeToken() 方法，支持自定义存储令牌
     */
    public static <AgentToken extends AccessToken, TicketTokenAgent extends TicketToken> void runThreadInitToken(String corpId, String agentId, String logPrevDesc, AgentToken agentToken, Integer residueSecond, Integer waitSecond, TicketTokenAgent ticketTokenAgent) {

        new Thread(new Runnable() {
            @Override
            public void run() {
                while (true) {
                    try {
                        InterfaceResult<JsapiTicketResForAgent> result = initToken(corpId, agentId, logPrevDesc, agentToken, ticketTokenAgent);
                        if (result.isSuccess()) { // 令牌获取成功
                            String ticketValue = result.getData().getTicket();
                            Integer expiresIn = result.getData().getExpiresIn();

                            logger.debug("企业应用ID: {}, 定时任务获取 应用 jsApi 令牌 成功，expiresIn = {}， ticket = {}", agentId, expiresIn, ticketValue);
                            Threads.sleep((expiresIn - (residueSecond == null ? 30 : residueSecond)) * 1000); // 休眠到 有效期剩余 30秒
                        } else { // 令牌获取失败
                            logger.error("企业应用ID: {}, 定时任务获取 应用 jsApi 令牌 失败, errCode:{} errMsg:{}", agentId, result.getCode(), result.getMsg());
                            Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                        }
                    } catch (Exception e) {
                        logger.error("企业应用ID: " + agentId + ", 定时任务获取 应用jsapi令牌 异常", e);
                        Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                    }
                }
            }
        }).start();
    }
    // ------------------------------- 令牌线程 end -------------------------------------------
}
