package cn.net.wanmo.common.weixin.work.inner.server_api.storage;

import cn.net.wanmo.common.result.InterfaceResult;
import cn.net.wanmo.common.util.DateUtil;
import cn.net.wanmo.common.util.Threads;
import cn.net.wanmo.common.weixin.work.inner.pojo.log.TokenLog;
import cn.net.wanmo.common.weixin.work.inner.pojo.token.AccessToken;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.token.AccessTokenReq;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.token.AccessTokenRes;
import cn.net.wanmo.common.weixin.work.inner.pojo.Corp;
import cn.net.wanmo.common.weixin.work.inner.server_api.util.WorkInnerUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 企业配置信息
 */
public class AccessTokenForCorp {
    private static Logger logger = LoggerFactory.getLogger(AccessTokenForCorp.class);


    // ------------------------------- 企业 配置 start -------------------------------------------
    /**
     * 用来存储企业配置信息 <br/>
     * key: 企业ID  <br/>
     * value: 企业配置信息  <br/>
     */
    private static Map<String, Corp> corpMap = new HashMap<>();

    /**
     * 设置企业相关配置
     *
     * @param corpId 企业ID
     * @param corp   企业配置
     */
    public static <AddressBookToken extends AccessToken> void putCorp(String corpId, Corp<AddressBookToken> corp) {
        corp.getAddressBookToken().storeToken(); // 执行自定义保存通讯录令牌
        corpMap.put(corpId, corp);
    }

    /**
     * 获取企业相关配置
     *
     * @param corpId 企业ID
     * @return 企业配置
     */
    public static <AddressBookToken extends AccessToken> Corp<AddressBookToken> getCorp(String corpId) {
        return corpMap.getOrDefault(corpId, new Corp<AddressBookToken>());
    }

    // ------------------------------- 企业 配置 start -------------------------------------------

    // ------------------------------- 令牌初始化 start -------------------------------------------

    /**
     * 获取通讯录令牌，并且保存令牌
     *
     * @param corp 企业
     * @return 接口实际响应的结果
     */
    public static InterfaceResult<AccessTokenRes> initTokenForAddressBook(Corp corp) {
        String corpId = corp.getCorpId();
        String addressBookSecret = corp.getAddressBookSecret();

        return initTokenForAddressBook(corpId, addressBookSecret, null);
    }

    /**
     * 获取通讯录令牌，并且保存令牌
     *
     * @param corp             企业
     * @param addressBookToken 扩展的自定义 token 对象，重写 storeToken() 方法，支持自定义存储令牌
     * @return 接口实际响应的结果
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<AccessTokenRes> initTokenForAddressBook(Corp corp, AddressBookToken addressBookToken) {
        String corpId = corp.getCorpId();
        String addressBookSecret = corp.getAddressBookSecret();

        return initTokenForAddressBook(corpId, addressBookSecret, addressBookToken);
    }

    /**
     * 获取通讯录令牌，并且保存令牌
     *
     * @param corpId            企业 ID
     * @param addressBookSecret 通讯录密钥
     * @return 接口实际响应的结果
     */
    public static InterfaceResult<AccessTokenRes> initTokenForAddressBook(String corpId, String addressBookSecret) {
        return initTokenForAddressBook(corpId, addressBookSecret, null);
    }

    /**
     * 获取通讯录令牌，并且保存令牌
     *
     * @param corpId            企业 ID
     * @param addressBookSecret 通讯录密钥
     * @param addressBookToken  扩展的自定义 token 对象，重写 storeToken() 方法，支持自定义存储令牌
     * @return 接口实际响应的结果
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<AccessTokenRes> initTokenForAddressBook(String corpId, String addressBookSecret, AddressBookToken addressBookToken) {
        AccessTokenReq req = AccessTokenReq.build(corpId, addressBookSecret);
        InterfaceResult<AccessTokenRes> result = WorkInnerUtil.getAccessToken(req);

        AccessToken accessToken = new AccessToken();
        accessToken.setCorpId(corpId);
        accessToken.setTokenType(AccessToken.TokenType.access_address_book);
        accessToken.setCode(result.getCode());
        accessToken.setMsg(result.getMsg());
        accessToken.setResTime(DateUtil.nowLong());

        if (result.isSuccess()) {
            AccessTokenRes data = result.getData();
            accessToken.setValue(data.getToken());
            accessToken.setExpiresIn(data.getExpiresIn());
            accessToken.setResTime(data.getResTime());
        }

        if (Objects.isNull(addressBookToken)) {
            Corp corp = getCorp(corpId);
            corp.setAddressBookToken(accessToken);
            putCorp(corpId, corp);

            logger.debug("企业微信ID: {}, 执行默认存储通讯录令牌完成 ...", corpId);
        } else {
            { // 赋值操作
                addressBookToken.setCorpId(corpId);
                addressBookToken.setTokenType(accessToken.getTokenType());
                addressBookToken.setCode(accessToken.getCode());
                addressBookToken.setMsg(accessToken.getMsg());
                addressBookToken.setValue(accessToken.getValue());
                addressBookToken.setExpiresIn(accessToken.getExpiresIn());
                addressBookToken.setResTime(accessToken.getResTime());
            }
            Corp<AddressBookToken> corp = getCorp(corpId);
            corp.setAddressBookToken(addressBookToken);
            putCorp(corpId, corp);

            logger.debug("企业微信ID: {}, 执行自定义存储通讯录令牌完成 ...", corpId);
        }

        { // 内存中记录日志
            TokenLog.add(corpId, null, accessToken.getTokenType().getLabel(), accessToken, TokenLog.addressBookTokens);
        }

        return result;
    }
    // ------------------------------- 令牌初始化 end -------------------------------------------

    // ------------------------------- 令牌线程 start -------------------------------------------

    /**
     * 自动定时获取 通讯录令牌
     *
     * @param corp 企业
     */
    public static void runThreadInitTokenForAddressBook(Corp corp) {
        String corpId = corp.getCorpId();
        String addressBookSecret = corp.getAddressBookSecret();

        runThreadInitTokenForAddressBook(corpId, addressBookSecret, null, null, null);
    }

    /**
     * 自动定时获取 通讯录令牌
     *
     * @param corp             企业
     * @param addressBookToken 扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <AddressBookToken extends AccessToken> void runThreadInitTokenForAddressBook(Corp corp, AddressBookToken addressBookToken) {
        String corpId = corp.getCorpId();
        String addressBookSecret = corp.getAddressBookSecret();

        runThreadInitTokenForAddressBook(corpId, addressBookSecret, null, null, addressBookToken);
    }

    /**
     * 自动定时获取 通讯录令牌
     *
     * @param corp          企业
     * @param residueSecond 成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond    失败后，等待多少时间后重新获取（默认 60秒）
     */
    public static <AddressBookToken extends AccessToken> void runThreadInitTokenForAddressBook(Corp corp, Integer residueSecond, Integer waitSecond) {
        String corpId = corp.getCorpId();
        String addressBookSecret = corp.getAddressBookSecret();

        runThreadInitTokenForAddressBook(corpId, addressBookSecret, residueSecond, waitSecond, null);
    }

    /**
     * 自动定时获取 通讯录令牌
     *
     * @param corp             企业
     * @param residueSecond    成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond       失败后，等待多少时间后重新获取（默认 60秒）
     * @param addressBookToken 扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <AddressBookToken extends AccessToken> void runThreadInitTokenForAddressBook(Corp corp, AddressBookToken addressBookToken, Integer residueSecond, Integer waitSecond) {
        String corpId = corp.getCorpId();
        String addressBookSecret = corp.getAddressBookSecret();

        runThreadInitTokenForAddressBook(corpId, addressBookSecret, residueSecond, waitSecond, addressBookToken);
    }

    /**
     * 自动定时获取 通讯录令牌
     *
     * @param corpId            企业 ID
     * @param addressBookSecret 通讯录密钥
     */
    public static void runThreadInitTokenForAddressBook(final String corpId, final String addressBookSecret) {
        runThreadInitTokenForAddressBook(corpId, addressBookSecret, null, null, null);
    }

    /**
     * 自动定时获取 通讯录令牌
     *
     * @param corpId            企业 ID
     * @param addressBookSecret 通讯录密钥
     * @param addressBookToken  扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <AddressBookToken extends AccessToken> void runThreadInitTokenForAddressBook(final String corpId, final String addressBookSecret, AddressBookToken addressBookToken) {
        runThreadInitTokenForAddressBook(corpId, addressBookSecret, null, null, addressBookToken);
    }

    /**
     * 自动定时获取 通讯录令牌
     *
     * @param corpId            企业 ID
     * @param addressBookSecret 通讯录密钥
     * @param residueSecond     成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond        失败后，等待多少时间后重新获取（默认 60秒）
     */
    public static void runThreadInitTokenForAddressBook(final String corpId, final String addressBookSecret, Integer residueSecond, Integer waitSecond) {
        runThreadInitTokenForAddressBook(corpId, addressBookSecret, residueSecond, waitSecond, null);
    }

    /**
     * 自动定时获取 通讯录令牌
     *
     * @param corpId            企业 ID
     * @param addressBookSecret 通讯录密钥
     * @param residueSecond     成功后，剩余多少时间后重新获取（默认 30秒）
     * @param waitSecond        失败后，等待多少时间后重新获取（默认 60秒）
     * @param addressBookToken  扩展的自定义 token 对象，重写 syncStore() 方法，支持自定义存储令牌
     */
    public static <AddressBookToken extends AccessToken> void runThreadInitTokenForAddressBook(final String corpId, final String addressBookSecret, Integer residueSecond, Integer waitSecond, AddressBookToken addressBookToken) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                while (true) {
                    try {
                        InterfaceResult<AccessTokenRes> result = initTokenForAddressBook(corpId, addressBookSecret, addressBookToken);
                        if (result.isSuccess()) { // 令牌获取成功
                            String token = result.getData().getToken();
                            Integer expiresIn = result.getData().getExpiresIn();

                            logger.debug("企业微信ID: {}, 定时任务获取 通讯录令牌 成功，expiresIn = {}， token = {}", corpId, expiresIn, token);
                            Threads.sleep((expiresIn - (residueSecond == null ? 30 : residueSecond)) * 1000); // 休眠到 有效期剩余 30秒
                        } else { // 令牌获取失败
                            logger.error("企业微信ID: {}, 定时任务获取 通讯录令牌 失败, errCode:{} errMsg:{}", corpId, result.getCode(), result.getMsg());
                            Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                        }
                    } catch (Exception e) {
                        logger.error("企业微信ID: " + corpId + ", 定时任务获取 通讯录令牌 异常", e);
                        Threads.sleep((waitSecond == null ? 60 : waitSecond) * 1000); // 休眠到 60秒
                    }
                }
            }
        }).start();
    }
    // ------------------------------- 令牌线程 end -------------------------------------------
}
