package cn.net.wanmo.common.weixin.work.inner.server_api.util.address_book;

import cn.net.wanmo.common.result.InterfaceResult;
import cn.net.wanmo.common.restful.SendUtil;
import cn.net.wanmo.common.util.StringUtil;
import cn.net.wanmo.common.weixin.work.inner.pojo.token.AccessToken;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.address_book.*;
import cn.net.wanmo.common.weixin.work.inner.pojo.Corp;
import cn.net.wanmo.common.weixin.work.inner.server_api.util.WorkInnerUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 企业内部开发 -> 服务端API -> 通讯录管理 -> 成员管理 <br/>
 * <pre>
 *     1. 电脑端企业微信 -> 安全与管理 -> 管理工具 -> 通讯录同步 -> 开启;
 *     2. 查看 Secret;
 *     3. 企业可信IP配置;
 * </pre>
 */
public class UserUtil {
    private static Logger logger = LoggerFactory.getLogger(UserUtil.class);

    /**
     * 创建成员
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserCreateRes> createUser(Corp<AddressBookToken> corp, UserCreateReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return createUser(logPrevDesc, token, req);
    }

    /**
     * 创建成员
     */
    public static InterfaceResult<UserCreateRes> createUser(String logPrevDesc, String token, UserCreateReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 读取成员: ");

        req.setBody(req.toJSONString());
        UserCreateRes res = new UserCreateRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/create?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * 读取成员 <br/>
     * 注： 通讯录查询接口停用
     *
     * @param corp 企业
     * @param req  请求体
     * @return 响应结果
     */
    @Deprecated
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserGetRes> getUser(Corp<AddressBookToken> corp, UserGetReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return getUser(logPrevDesc, token, req);
    }

    /**
     * 读取成员 <br/>
     * 注： 通讯录查询接口停用
     *
     * @param logPrevDesc 日志前缀描述
     * @param token       通讯录令牌
     * @param req         请求体
     * @return 响应结果
     */
    @Deprecated
    public static InterfaceResult<UserGetRes> getUser(String logPrevDesc, String token, UserGetReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 读取成员: ");

        req.setBody(req.toJSONString());
        UserGetRes res = new UserGetRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/get?access_token=ACCESS_TOKEN&userid=USERID";
        url = url.replace("ACCESS_TOKEN", token).replace("USERID", req.getUserId());

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }


    /**
     * 更新成员
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserUpdateRes> updateUser(Corp<AddressBookToken> corp, UserUpdateReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return updateUser(logPrevDesc, token, req);
    }

    /**
     * 更新成员
     */
    public static InterfaceResult<UserUpdateRes> updateUser(String logPrevDesc, String token, UserUpdateReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 更新成员: ");

        req.setBody(req.toJSONString());
        UserUpdateRes res = new UserUpdateRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/update?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * 删除成员
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserDeleteRes> deleteUser(Corp<AddressBookToken> corp, UserDeleteReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return deleteUser(logPrevDesc, token, req);
    }

    /**
     * 删除成员
     */
    public static InterfaceResult<UserDeleteRes> deleteUser(String logPrevDesc, String token, UserDeleteReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 删除成员: ");

        UserDeleteRes res = new UserDeleteRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/delete?access_token=ACCESS_TOKEN&userid=USERID";
        url = url.replace("ACCESS_TOKEN", token);
        url = url.replace("USERID", StringUtil.isBlank(req.getUserid()) ? "" : req.getUserid());

        return SendUtil.sendGet(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * 批量删除成员
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserBatchDeleteRes> batchDeleteUser(Corp<AddressBookToken> corp, UserBatchDeleteReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return batchDeleteUser(logPrevDesc, token, req);
    }

    /**
     * 批量删除成员
     */
    public static InterfaceResult<UserBatchDeleteRes> batchDeleteUser(String logPrevDesc, String token, UserBatchDeleteReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 批量删除成员: ");

        req.setBody(req.toJSONString());
        UserBatchDeleteRes res = new UserBatchDeleteRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/batchdelete?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * 获取部门成员<br/>
     * 注：已停用
     */
    public static <AddressBookToken extends AccessToken> void simpleListByDeptId() {
        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/simplelist?access_token=ACCESS_TOKEN&department_id=DEPARTMENT_ID";
    }

    /**
     * 获取部门成员详情<br/>
     * * 注：已停用
     */
    public static <AddressBookToken extends AccessToken> void listByDeptId() {
        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/list?access_token=ACCESS_TOKEN&department_id=DEPARTMENT_ID";
    }

    /**
     * userid转openid
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserConvertToOpenidRes> convertToOpenid(Corp<AddressBookToken> corp, UserConvertToOpenidReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return convertToOpenid(logPrevDesc, token, req);
    }

    /**
     * userid转openid
     */
    public static InterfaceResult<UserConvertToOpenidRes> convertToOpenid(String logPrevDesc, String token, UserConvertToOpenidReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 userid转openid: ");

        req.setBody(req.toJSONString());
        UserConvertToOpenidRes res = new UserConvertToOpenidRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/convert_to_openid?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * openid转userid
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserConvertToUseridRes> convertToUserid(Corp<AddressBookToken> corp, UserConvertToUseridReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return convertToUserid(logPrevDesc, token, req);
    }

    /**
     * openid转userid
     */
    public static InterfaceResult<UserConvertToUseridRes> convertToUserid(String logPrevDesc, String token, UserConvertToUseridReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 openid转userid: ");

        req.setBody(req.toJSONString());
        UserConvertToUseridRes res = new UserConvertToUseridRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/convert_to_userid?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * 二次验证
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserAuthSuccRes> authSucc(Corp<AddressBookToken> corp, UserAuthSuccReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return authSucc(logPrevDesc, token, req);
    }

    /**
     * 二次验证
     */
    public static InterfaceResult<UserAuthSuccRes> authSucc(String logPrevDesc, String token, UserAuthSuccReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 登录二次验证: ");

        UserAuthSuccRes res = new UserAuthSuccRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/authsucc?access_token=ACCESS_TOKEN&userid=USERID";
        url = url.replace("ACCESS_TOKEN", token);
        url = url.replace("USERID", StringUtil.isBlank(req.getUserid()) ? "" : req.getUserid());

        return SendUtil.sendGet(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * 邀请成员<br/>
     * 企业可通过接口批量邀请成员使用企业微信，邀请后将通过短信或邮件下发通知。
     */
    public static <AddressBookToken extends AccessToken> void batchInvite() {
        String url = "https://qyapi.weixin.qq.com/cgi-bin/batch/invite?access_token=ACCESS_TOKEN";
        // todo 邀请成员
    }

    /**
     * 获取加入企业二维码<br/>
     * 支持企业用户获取实时成员加入二维码。
     */
    public static <AddressBookToken extends AccessToken> void getJoinQrcode() {
        String url = "https://qyapi.weixin.qq.com/cgi-bin/corp/get_join_qrcode?access_token=ACCESS_TOKEN&size_type=SIZE_TYPE";
        // todo 获取加入企业二维码
    }


    /**
     * 手机号获取userid<br/>
     * 通过手机号获取其所对应的userid。 <br/>
     * 注：请确保手机号的正确性，若出错的次数超出企业规模人数的20%，会导致1天不可调用。
     *
     * @param corp 企业
     * @param req  请求体
     * @return 响应结果
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserGetIdByMobileRes> getUseridByMobile(Corp<AddressBookToken> corp, UserGetIdByMobileReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return getUseridByMobile(logPrevDesc, token, req);
    }


    /**
     * 手机号获取userid<br/>
     * 通过手机号获取其所对应的userid。 <br/>
     * 注：请确保手机号的正确性，若出错的次数超出企业规模人数的20%，会导致1天不可调用。
     *
     * @param logPrevDesc 日志前缀描述
     * @param token       通讯录令牌
     * @param req         请求体
     * @return 响应结果
     */
    public static InterfaceResult<UserGetIdByMobileRes> getUseridByMobile(String logPrevDesc, String token, UserGetIdByMobileReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 手机号获取userid: ");

        req.setBody(req.toJSONString());
        UserGetIdByMobileRes res = new UserGetIdByMobileRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/getuserid?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }


    /**
     * 邮箱获取userid <br/>
     * 通过邮箱获取其所对应的userid。 <br/>
     * 注：请确保邮箱的正确性，若出错的次数较多，会导致1天不可调用。（限制阈值取决于安装企业的员工规模）
     *
     * @param corp 企业
     * @param req  请求体
     * @return 响应结果
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserGetIdByEmailRes> getUseridByEmail(Corp<AddressBookToken> corp, UserGetIdByEmailReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return getUseridByEmail(logPrevDesc, token, req);
    }

    /**
     * 邮箱获取userid <br/>
     * 通过邮箱获取其所对应的userid。 <br/>
     * 注：请确保邮箱的正确性，若出错的次数较多，会导致1天不可调用。（限制阈值取决于安装企业的员工规模）
     *
     * @param logPrevDesc 日志前缀描述
     * @param token       通讯录令牌
     * @param req         请求体
     * @return 响应结果
     */
    public static InterfaceResult<UserGetIdByEmailRes> getUseridByEmail(String logPrevDesc, String token, UserGetIdByEmailReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 邮箱获取userid: ");

        req.setBody(req.toJSONString());
        UserGetIdByEmailRes res = new UserGetIdByEmailRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/get_userid_by_email?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * 获取成员ID列表<br/>
     * 获取企业成员的userid与对应的部门ID列表<br/>
     * 注：仅支持通过“通讯录同步secret”调用。<br/>
     *
     * @param corp 企业
     * @param req  请求体
     * @return 响应结果
     */
    public static <AddressBookToken extends AccessToken> InterfaceResult<UserListIdRes> listId(Corp<AddressBookToken> corp, UserListIdReq req) {
        String logPrevDesc = corp.getLogPrevDesc();
        String token = corp.takeTokenForAddressBook();

        return listId(logPrevDesc, token, req);
    }


    /**
     * 获取成员ID列表<br/>
     * 获取企业成员的userid与对应的部门ID列表<br/>
     * 注：仅支持通过“通讯录同步secret”调用。<br/>
     *
     * @param token 通讯录令牌
     * @param req   请求体
     * @return 响应结果
     */
    public static InterfaceResult<UserListIdRes> listId(String token, UserListIdReq req) {
        return listId(null, token, req);
    }

    /**
     * 获取成员ID列表<br/>
     * 获取企业成员的userid与对应的部门ID列表<br/>
     * 注：仅支持通过“通讯录同步secret”调用。<br/>
     *
     * @param logPrevDesc 日志前缀描述
     * @param token       通讯录令牌
     * @param req         请求体
     * @return 响应结果
     */
    public static InterfaceResult<UserListIdRes> listId(String logPrevDesc, String token, UserListIdReq req) {
        String msgPre = WorkInnerUtil.getMsgPre(logPrevDesc, "通讯录管理 获取成员ID列表: ");

        req.setBody(req.toJSONString());
        UserListIdRes res = new UserListIdRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/user/list_id?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }
}
