package cn.net.wanmo.common.weixin.work.inner.server_api.util.message_push;

import cn.net.wanmo.common.result.InterfaceResult;
import cn.net.wanmo.common.restful.SendUtil;
import cn.net.wanmo.common.weixin.work.inner.pojo.Agent;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.message_push.app.MessageReq;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.message_push.app.MessageRes;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.message_recall.MessageRecallReq;
import cn.net.wanmo.common.weixin.work.inner.server_api.pojo.message_recall.MessageRecallRes;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 企业微信开放了消息发送接口，企业可以使用这些接口让自定义应用与企业微信后台或用户间进行双向通信。
 */
public class MessageUtil {
    private static Logger logger = LoggerFactory.getLogger(MessageUtil.class);

    /**
     * 应用支持推送文本、图片、视频、文件、图文等类型。 <br/>
     *
     * @param agent 应用配置
     * @param req   请求体
     * @return 响应结果
     */
    public static InterfaceResult<MessageRes> sendMessage(Agent agent, MessageReq req) {
        String logPrevDesc = agent.getLogPrevDesc();
        String token = agent.takeToken();

        return sendMessage(logPrevDesc, token, req);
    }

    /**
     * 应用支持推送文本、图片、视频、文件、图文等类型。 <br/>
     *
     * @param token 应用令牌
     * @param req   请求体
     * @return 响应结果
     */
    public static InterfaceResult<MessageRes> sendMessage(String token, MessageReq req) {
        return sendMessage(null, token, req);
    }

    /**
     * 应用支持推送文本、图片、视频、文件、图文等类型。 <br/>
     *
     * @param logPrevDesc 日志前缀描述
     * @param token       应用令牌
     * @param req         请求体
     * @return 响应结果
     */
    public static InterfaceResult<MessageRes> sendMessage(String logPrevDesc, String token, MessageReq req) {
        String msgPre = (StringUtils.isBlank(logPrevDesc) ? "" : (logPrevDesc + ":")) + "消息推送 发送应用消息: ";

        req.setBody(req.toJSONString());
        MessageRes res = new MessageRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/message/send?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }

    /**
     * 本接口可以撤回24小时内通过发送应用消息接口推送的消息，仅可撤回企业微信端的数据，微信插件端的数据不支持撤回。 <br/>
     *
     * @param agent 应用配置
     * @param req   请求体
     * @return 响应结果
     */
    public static InterfaceResult<MessageRecallRes> recallMessage(Agent agent, MessageRecallReq req) {
        String logPrevDesc = agent.getLogPrevDesc();
        String token = agent.takeToken();

        return recallMessage(logPrevDesc, token, req);
    }

    /**
     * 本接口可以撤回24小时内通过发送应用消息接口推送的消息，仅可撤回企业微信端的数据，微信插件端的数据不支持撤回。 <br/>
     *
     * @param token 应用令牌
     * @param msgId 消息ID
     * @return 响应结果
     */
    public static InterfaceResult<MessageRecallRes> recallMessage(String token, String msgId) {
        return recallMessage(null, token, msgId);
    }

    /**
     * 本接口可以撤回24小时内通过发送应用消息接口推送的消息，仅可撤回企业微信端的数据，微信插件端的数据不支持撤回。 <br/>
     *
     * @param logPrevDesc 日志前缀描述
     * @param token       应用令牌
     * @param msgId       消息ID
     * @return 响应结果
     */
    public static InterfaceResult<MessageRecallRes> recallMessage(String logPrevDesc, String token, String msgId) {
        MessageRecallReq req = MessageRecallReq.build(msgId);
        return recallMessage(logPrevDesc, token, req);
    }

    /**
     * 本接口可以撤回24小时内通过发送应用消息接口推送的消息，仅可撤回企业微信端的数据，微信插件端的数据不支持撤回。 <br/>
     *
     * @param token 应用令牌
     * @param req   请求体
     * @return 响应结果
     */
    public static InterfaceResult<MessageRecallRes> recallMessage(String token, MessageRecallReq req) {
        return recallMessage(null, token, req);
    }

    /**
     * 本接口可以撤回24小时内通过发送应用消息接口推送的消息，仅可撤回企业微信端的数据，微信插件端的数据不支持撤回。 <br/>
     *
     * @param logPrevDesc 日志前缀描述
     * @param token       应用令牌
     * @param req         请求体
     * @return 响应结果
     */
    public static InterfaceResult<MessageRecallRes> recallMessage(String logPrevDesc, String token, MessageRecallReq req) {
        String msgPre = (StringUtils.isBlank(logPrevDesc) ? "" : (logPrevDesc + ":")) + "消息撤回 撤回应用消息: ";

        req.setBody(req.toJSONString());
        MessageRecallRes res = new MessageRecallRes();

        String url = "https://qyapi.weixin.qq.com/cgi-bin/message/recall?access_token=ACCESS_TOKEN";
        url = url.replace("ACCESS_TOKEN", token);

        return SendUtil.sendPostForJson(msgPre, url, req, res, logger, Integer.class);
    }
}
