package com.pig4cloud.pigx.common.email;

import com.pig4cloud.pigx.common.email.core.EmailTemplate;
import com.pig4cloud.pigx.common.email.core.ToEmail;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.FileSystemResource;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSender;
import org.springframework.mail.javamail.MimeMessageHelper;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;
import java.io.File;
import java.util.Arrays;

@Slf4j
@RequiredArgsConstructor
public class DefaultEmailTemplate implements EmailTemplate {

	private final JavaMailSender mailSender;

	@Value("${spring.mail.username}")
	private String from;

	/**
	 * 简单邮件
	 * @param toEmail
	 * @return
	 */
	public Boolean commonEmail(ToEmail toEmail) {
		// 创建简单邮件消息
		SimpleMailMessage message = new SimpleMailMessage();
		// 谁发的
		message.setFrom(from);
		// 谁要接收
		message.setTo(toEmail.getTos());
		// 邮件标题
		message.setSubject(toEmail.getSubject());
		// 邮件内容
		message.setText(toEmail.getContent());

		mailSender.send(message);
		log.info("邮件发送成功 -》 {}", Arrays.asList(toEmail.getTos()));

		return Boolean.TRUE;

	}

	/**
	 * html邮件
	 * @param toEmail
	 * @return
	 * @throws MessagingException
	 */
	public Boolean htmlEmail(ToEmail toEmail) throws MessagingException {
		// 创建一个MINE消息
		MimeMessage message = mailSender.createMimeMessage();
		MimeMessageHelper minehelper = new MimeMessageHelper(message, true);
		// 谁发
		minehelper.setFrom(from);
		// 谁要接收
		minehelper.setTo(toEmail.getTos());
		// 邮件主题
		minehelper.setSubject(toEmail.getSubject());
		// 邮件内容 true 表示带有附件或html
		minehelper.setText(toEmail.getContent(), true);

		mailSender.send(message);
		log.info("邮件发送成功 -》 {}", Arrays.asList(toEmail.getTos()));
		return Boolean.TRUE;

	}

	/**
	 * 静态资源文件
	 * @param toEmail
	 * @param file
	 * @param resId
	 * @return
	 */
	@SneakyThrows
	public Boolean staticEmail(ToEmail toEmail, File file, String resId) {
		// 创建一个MINE消息
		MimeMessage message = mailSender.createMimeMessage();
		MimeMessageHelper helper = new MimeMessageHelper(message, true);
		// 谁发
		helper.setFrom(from);
		// 谁接收
		helper.setTo(toEmail.getTos());
		// 邮件主题
		helper.setSubject(toEmail.getSubject());
		// 邮件内容 true 表示带有附件或html
		// 邮件内容拼接
		String content = "<html><body><img width='250px' src='cid:" + resId + "'>" + toEmail.getContent()
				+ "</body></html>";
		helper.setText(content, true);
		// 蒋 multpartfile 转为file
		FileSystemResource res = new FileSystemResource(file);

		// 添加内联资源，一个id对应一个资源，最终通过id来找到该资源
		helper.addInline(resId, res);
		mailSender.send(message);
		log.info("邮件发送成功 -》 {}", Arrays.asList(toEmail.getTos()));
		return Boolean.TRUE;
	}

	/**
	 * 带有附件的邮件
	 * @param toEmail
	 * @param file
	 * @return
	 */
	@SneakyThrows
	public Boolean enclosureEmail(ToEmail toEmail, File file) {
		// 创建一个MINE消息
		MimeMessage message = mailSender.createMimeMessage();
		MimeMessageHelper helper = new MimeMessageHelper(message, true);
		// 谁发
		helper.setFrom(from);
		// 谁接收
		helper.setTo(toEmail.getTos());
		// 邮件主题
		helper.setSubject(toEmail.getSubject());
		// 邮件内容 true 表示带有附件或html
		helper.setText(toEmail.getContent(), true);
		FileSystemResource fileSystemResource = new FileSystemResource(file);
		String filename = fileSystemResource.getFilename();
		// 添加附件
		helper.addAttachment(filename, file);
		mailSender.send(message);
		log.info("邮件发送成功 -》 {}", Arrays.asList(toEmail.getTos()));
		return Boolean.TRUE;

	}

}
