package cn.crushes.cloud.core.pay.alipay;

import cn.crushes.cloud.core.common.exception.PayException;
import com.alipay.api.AlipayApiException;
import com.alipay.api.AlipayClient;
import com.alipay.api.CertAlipayRequest;
import com.alipay.api.DefaultAlipayClient;
import com.alipay.api.internal.util.file.IOUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.context.properties.PropertyMapper;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.ClassPathResource;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

/**
 * @author youta
 */
@Slf4j
@Configuration
@ConditionalOnProperty(prefix = "ali.pay", name = "v1.app-id")
@EnableConfigurationProperties(AliPayProperties.class)
public class AliPayConfiguration {

    @Bean
    public AlipayClient alipayClient(AliPayProperties aliPayProperties) throws AlipayApiException {

        PropertyMapper propertyMapper = PropertyMapper.get().alwaysApplyingWhenNonNull();

        AliPayProperties.V1 v1 = aliPayProperties.getV1();
        CertAlipayRequest certAlipayRequest = new CertAlipayRequest();
        propertyMapper.from(v1::getServerUrl).to(certAlipayRequest::setServerUrl);
        propertyMapper.from(v1::getAppId).to(certAlipayRequest::setAppId);
        propertyMapper.from(v1::getAppPrivateKeyPath).as(this::appRSAPrivateKey).to(certAlipayRequest::setPrivateKey);
        propertyMapper.from(v1::getFormat).to(certAlipayRequest::setFormat);
        propertyMapper.from(v1::getCharset).to(certAlipayRequest::setCharset);
        propertyMapper.from(v1::getSignType).to(certAlipayRequest::setSignType);
        propertyMapper.from(v1::getAppCertPublicKeyPath).as(this::getContentFromClassPath).to(certAlipayRequest::setCertContent);
        propertyMapper.from(v1::getAlipayPublicCertPath).as(this::getContentFromClassPath).to(certAlipayRequest::setAlipayPublicCertContent);
        propertyMapper.from(v1::getAlipayRootCertPath).as(this::getContentFromClassPath).to(certAlipayRequest::setRootCertContent);
        return new DefaultAlipayClient(certAlipayRequest);
    }


    private String getContentFromClassPath(String classPath) {
        ClassPathResource resource = new ClassPathResource(classPath);
        try (InputStreamReader in = new InputStreamReader(resource.getInputStream())) {
            return IOUtils.toString(in);
        } catch (IOException e) {
            log.error("ali pay root cert is invalid ,{}", e.getMessage());
            throw new PayException("ali pay root cert path is invalid");
        }
    }


    private String appRSAPrivateKey(String classPath) {
        ClassPathResource resource = new ClassPathResource(classPath);
        try (InputStreamReader inputStreamReader = new InputStreamReader(resource.getInputStream())) {
            try (BufferedReader bufferedReader = new BufferedReader(inputStreamReader)) {
                return bufferedReader.readLine();
            }
        } catch (IOException e) {
            log.error("ali pay app private key is required ,{}", e.getMessage());
            throw new PayException("ali pay app private key is required");
        }
    }

}
