package cn.crushes.cloud.core.pay.wechat.v3;

import cn.crushes.cloud.core.pay.wechat.v2.WechatAllocationApi;
import cn.crushes.cloud.core.pay.wechat.v2.WechatPayRedPackApi;
import cn.crushes.cloud.core.pay.wechat.v2.WechatPayTransfersApi;
import cn.crushes.cloud.core.pay.wechat.v2.WechatV2Client;

/**
 * 微信支付工具.
 *
 * @author youta
 */
public class WechatApiProvider {
    /**
     * 微信支付客户端.
     */
    private final WechatPayClient wechatPayClient;

    /**
     * Instantiates a new Wechat api provider.
     *
     * @param wechatPayClient the wechat pay client
     */
    public WechatApiProvider(WechatPayClient wechatPayClient) {
        this.wechatPayClient = wechatPayClient;
    }

    /**
     * 代金券.
     *
     * @param tenantId the tenant id
     * @return the wechat marketing favor api
     */
    public WechatMarketingFavorApi favorApi(String tenantId) {
        return new WechatMarketingFavorApi(this.wechatPayClient, tenantId);
    }

    /**
     * 普通支付-直连模式.
     *
     * @param tenantId the tenant id
     * @return the wechat pay api
     */
    public WechatDirectPayApi directPayApi(String tenantId) {
        return new WechatDirectPayApi(wechatPayClient, tenantId);
    }


    /**
     * 普通支付-服务商模式.
     *
     * @param tenantId the tenant id
     * @return the wechat pay api
     */
    public WechatPartnerPayApi partnerPayApi(String tenantId) {
        return new WechatPartnerPayApi(wechatPayClient, tenantId);
    }

    /**
     * 合单支付.
     *
     * @param tenantId the tenant id
     * @return the wechat combine pay api
     */
    public WechatCombinePayApi combinePayApi(String tenantId) {
        return new WechatCombinePayApi(wechatPayClient, tenantId);
    }

    /**
     * 微信支付分.
     *
     * @param tenantId the tenant id
     * @return the wechat pay score api
     */
    public WechatPayScoreApi payScoreApi(String tenantId) {
        return new WechatPayScoreApi(wechatPayClient, tenantId);
    }

    /**
     * 微信支付先享卡.
     *
     * @param tenantId the tenant id
     * @return the wechat discount card api
     */
    public WechatDiscountCardApi discountCardApi(String tenantId) {
        return new WechatDiscountCardApi(wechatPayClient, tenantId);
    }

    /**
     * 微信支付商家券.
     *
     * @param tenantId the tenant id
     * @return the wechat discount card api
     */
    public WechatMarketingBusiFavorApi busiFavorApi(String tenantId) {
        return new WechatMarketingBusiFavorApi(wechatPayClient, tenantId);
    }

    /**
     *  批量转账到零钱.
     * <p>
     * 批量转账到零钱提供商户同时向多个用户微信零钱转账的能力。商户可以使用批量转账到零钱用于费用报销、员工福利发放、合作伙伴货款或服务款项支付等场景，提高转账效率。
     *
     * @param tenantId the tenant id
     * @return the WechatBatchTransferApi
     */
    public WechatBatchTransferApi batchTransferApi(String tenantId) {
        return new WechatBatchTransferApi(wechatPayClient, tenantId);
    }

    /**
     * 回调.
     * <p>
     * 需要处理白名单、幂等性问题。
     *
     * @param tenantId the tenant id
     * @return the wechat pay callback
     */
    public WechatPayCallback callback(String tenantId) {
        return new WechatPayCallback(wechatPayClient.signatureProvider(), tenantId);
    }

    /**
     * 现金红包，基于V2
     *
     * @param tenantId the tenant id
     * @return wechat pay redpack api
     */
    public WechatPayRedPackApi redpackApi(String tenantId) {
        WechatMetaBean wechatMeta = wechatPayClient.signatureProvider()
                .wechatMetaContainer()
                .getWechatMeta(tenantId);
        WechatV2Client wechatV2Client = new WechatV2Client(wechatMeta);
        return new WechatPayRedPackApi(wechatV2Client);
    }

    /**
     * 企业付款到零钱，目前不包括到银行卡，基于V2
     *
     * @param tenantId the tenant id
     * @return wechat pay redpack api
     */
    public WechatPayTransfersApi transfersApi(String tenantId) {
        WechatMetaBean wechatMeta = wechatPayClient.signatureProvider()
                .wechatMetaContainer()
                .getWechatMeta(tenantId);
        WechatV2Client wechatV2Client = new WechatV2Client(wechatMeta);
        return new WechatPayTransfersApi(wechatV2Client);
    }

    /**
     * 微信支付分账，基于V2
     *
     * @param tenantId the tenant id
     * @return wechat allocation api
     */
    public WechatAllocationApi allocationApi(String tenantId) {
        WechatMetaBean wechatMeta = wechatPayClient.signatureProvider()
                .wechatMetaContainer()
                .getWechatMeta(tenantId);
        WechatV2Client wechatV2Client = new WechatV2Client(wechatMeta);
        return new WechatAllocationApi(wechatV2Client);
    }

}
