package cn.duskykite.open.call;

import cn.duskykite.open.call.path.LiteralPathResolver;
import cn.duskykite.open.call.path.PathResolver;
import cn.duskykite.open.call.path.QueryPathResolver;
import com.google.common.collect.ImmutableList;
import lombok.NonNull;
import org.apache.hc.core5.net.URIBuilder;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

/**
 * 路径对象
 * @author <a href="mailto:wh1zper@qq.com">wh1zper</a>
 */
public record Path(
        // 路径处理对象（多个）
        @NonNull Collection<? extends PathResolver> resolvers
) {

    /**
     * @return 路径对象构建对象
     */
    public static @NonNull Builder builder() {
        return new Builder();
    }

    /**
     * @param resolvers 路径处理对象（多个）
     */
    public Path {
        resolvers = ImmutableList.copyOf(resolvers);
    }

    /**
     * @param path 路径字符串
     */
    public Path(@NonNull String path) {
        this(List.of(new LiteralPathResolver(path)));
    }

    /**
     * 根据参数（多个）对路径中的变量进行替换，获取最终的路径字符串
     * @param variables 参数（多个）
     * @return 路径字符串
     */
    public @NonNull String resolve(Collection<@org.checkerframework.checker.nullness.qual.Nullable String> variables) {
        var builder = new URIBuilder();
        @Nullable var iterator = Optional.ofNullable(variables).filter($variables -> !$variables.isEmpty())
                .map(Collection::iterator).orElse(null);
        resolvers.forEach(resolver -> resolver.resolve(builder, iterator));
        return builder.toString();
    }

    @Override
    public @NonNull String toString() {
        return resolve(null);
    }

    /**
     * 路径对象构建对象
     * @param resolvers 路径处理对象（多个）
     */
    public record Builder(
            @NonNull Collection<PathResolver> resolvers
    ) {

        /**
         * constructor
         */
        public Builder() {
            this(new LinkedList<>());
        }

        /**
         * 构建路径对象
         * @return 路径对象
         */
        public @NonNull Path build() {
            return new Path(resolvers);
        }

        /**
         * 追加路径处理对象
         * @param resolver 路径处理对象
         * @return 当前操作的构建对象
         */
        public @NonNull Builder append(@NonNull PathResolver resolver) {
            resolvers.add(resolver);
            return this;
        }

        /**
         * 追加路径字符串
         * @param literal 路径字符串
         * @return 当前操作的构建对象
         */
        public @NonNull Builder appendLiteral(@NonNull String literal) {
            return append(new LiteralPathResolver(literal));
        }

        /**
         * 追加路径变量
         * @param variable 路径变量
         * @return 当前操作的构建对象
         */
        public @NonNull Builder appendPathVariable(@NonNull String variable) {
            return append((QueryPathResolver) () -> variable);
        }

        /**
         * 追加查询参数
         * @param name 查询参数名
         * @return 当前操作的构建对象
         */
        public @NonNull Builder appendQuery(@NonNull String name) {
            return append((QueryPathResolver) () -> name);
        }
    }
}
