package cn.duskykite.open.call;

import cn.duskykite.open.GsonSupport;
import cn.duskykite.open.call.requestbody.MultipartBodySupplier;
import cn.duskykite.open.call.result.data.PageIterator;
import cn.duskykite.open.call.result.mapper.JsonResultMapper;
import com.google.gson.reflect.TypeToken;
import lombok.AllArgsConstructor;
import lombok.NonNull;
import okhttp3.*;

import javax.annotation.Nullable;
import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.UnaryOperator;
import java.util.stream.Stream;

/**
 * 飞书开放接口远程调用对象
 * @author <a href="mailto:wh1zper@qq.com">wh1zper</a>
 */
@AllArgsConstructor
public class SAO {

    /**
     * 飞书开放接口主链接
     */
    @NonNull
    public static final HttpUrl OPENAPI_PRIMARY_URL =
            Objects.requireNonNull(HttpUrl.parse("https://open.feishu.cn/open-apis"));

    /**
     * 内容类型：JSON（UTF-8）
     */
    @NonNull
    public static final MediaType APPLICATION_JSON_UTF_8 = Objects.requireNonNull(
            MediaType.parse("application/json; charset=utf-8"));

    /**
     * JSON内容类型头部信息
     */
    @NonNull
    public static final Headers JSON_CONTENT_TYPE_HEADERS = toContentTypeHeaders(APPLICATION_JSON_UTF_8);

    /**
     * 生成授权信息
     * @param token 凭据信息
     * @return 授权信息
     */
    public static @NonNull String toAuthorization(@NonNull String token) {
        return "Bearer " + token;
    }

    /**
     * 生成内容类型的头部信息
     * @param mediaType 内容类型
     * @return 内容类型的头部信息
     */
    public static @NonNull Headers toContentTypeHeaders(@NonNull MediaType mediaType) {
        return Headers.of("Content-Type", mediaType.toString());
    }

    /**
     * 生成用于鉴权的头部信息
     * @param authorization 鉴权凭据信息
     * @return 鉴权的头部信息
     */
    public static @NonNull Headers toAuthorizationHeaders(@NonNull String authorization) {
        return Headers.of("Authorization", authorization);
    }

    /**
     * 分页式获取所有页的元素，并收集到统一的容器中
     * @param paging 根据分页标记执行分页式元素获取的方法
     * @param identity 统一的容器
     * @param accumulator 将分页元素统一化收集到容器中的方法
     * @return 已经收集完所有页的元素的容器
     * @param <P> 分页迭代对象的类型
     * @param <C> 统一化容器的类型
     */
    public static <P extends PageIterator, C> C pagingAll(
            @NonNull Function<String, ? extends P> paging,
            C identity,
            @NonNull BiFunction<C, ? super P, C> accumulator
    ) {
        @Nullable String pageToken = null;
        do {
            @Nullable var page = paging.apply(pageToken);
            accumulator.apply(identity, page);
            pageToken = Optional.ofNullable(page).map(PageIterator::$pageToken).orElse(null);
        } while (Objects.nonNull(pageToken));
        return identity;
    }

    /**
     * 开放接口主链接，例如<a href="https://open.feishu.cn/open-apis">飞书开放平台接口</a>
     */
    @NonNull
    private final HttpUrl primaryURL;

    /**
     * HTTP请求客户端
     */
    @NonNull
    private final OkHttpClient client;

    /**
     * 生成路径中的变量对应的字符串的列表
     * @param variables 路径中的变量数组
     * @return 路径中的变量字符串列表
     */
    protected static @NonNull List<@org.checkerframework.checker.nullness.qual.Nullable String> variables(
            Object... variables
    ) {
        return Optional.ofNullable(variables).filter($variables -> $variables.length > 0)
                .map($variables ->
                        Stream.of($variables).map(variable ->
                                        Optional.ofNullable(variable).map(Object::toString).orElse(null))
                                .toList())
                .orElseGet(List::of);
    }

    /**
     * 合成头部信息
     * @param headers 待合成的头部信息数组
     * @return 合成后的头部信息
     */
    protected static Headers headers(Headers... headers) {
        return Optional.ofNullable(headers).filter($headers -> $headers.length > 0)
                .map($headers -> {
                    if ($headers.length == 1) {
                        return $headers[0];
                    }

                    return Stream.of($headers).collect(
                            Headers.Builder::new,
                            Headers.Builder::addAll,
                            (builder, other) -> builder.addAll(other.build())).build();
                })
                .orElse(null);
    }

    /**
     * 生成Json格式的请求体
     * @param body 请求体对象
     * @return Json请求体
     */
    protected static @NonNull RequestBody toJsonRequestBody(@NonNull Object body) {
        return RequestBody.create(GsonSupport.toJson(body), APPLICATION_JSON_UTF_8);
    }

    /**
     * 生成复合请求体
     * @param supplier 复合请求体提供方法
     * @return 复合请求体
     */
    protected static @NonNull MultipartBody toMultipartBody(@NonNull MultipartBodySupplier supplier) {
        return supplier.get();
    }

    /**
     * 生成按Json格式转化为Java对象的转化方法
     * @param typeToken 类凭据
     * @return 按Json格式转化为Java对象的转化方法
     * @param <T> Java对象的类型
     */
    protected static <T> @NonNull JsonResultMapper<T> toJsonResultMapper(@NonNull TypeToken<? extends T> typeToken) {
        return jsonText -> GsonSupport.fromJson(jsonText, typeToken);
    }

    /**
     * 执行HTTP请求
     * @param method 请求方法
     * @param path 路径
     * @param headers 头部信息
     * @param resultMapper 结果转化方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    protected <T> @NonNull Response<T> request(
            @NonNull UnaryOperator<Request.Builder> method,
            @NonNull String path,
            Headers headers,
            @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        var builder = method.apply(new Request.Builder());
        if (Objects.nonNull(headers) && headers.size() > 0) {
            builder = builder.headers(headers);
        }
        var request = builder.url(primaryURL + path).build();
        var call = client.newCall(request);
        try (var response = call.execute()) {
            @Nullable var body = response.body();
            @Nullable T result;
            try (var ignored = (Closeable) () -> Optional.ofNullable(body).ifPresent(ResponseBody::close)) {
                result = resultMapper.apply(body);
            }
            return new Response<>(response.code(), response.headers(), result);
        } catch (IOException e) {
            throw new RuntimeException("execute http call exception: ", e);
        }
    }

    /**
     * 执行HTTP GET请求
     * @param path 路径
     * @param headers 头部信息
     * @param resultMapper 结果生成方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    protected <T> @NonNull Response<T> get(
            @NonNull String path, Headers headers, @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        return request(Request.Builder::get, path, headers, resultMapper);
    }

    /**
     * 执行HTTP POST请求
     * @param path 路径
     * @param headers 头部信息
     * @param body 请求体
     * @param resultMapper 结果生成方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    protected <T> @NonNull Response<T> post(
            @NonNull String path,
            Headers headers,
            @NonNull RequestBody body,
            @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        return request(builder -> builder.post(body), path, headers, resultMapper);
    }

    /**
     * 执行HTTP PATCH请求
     * @param path 路径
     * @param headers 头部信息
     * @param body 请求体
     * @param resultMapper 结果生成方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    protected <T> @NonNull Response<T> patch(
            @NonNull String path,
            Headers headers,
            @NonNull RequestBody body,
            @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        return request(builder -> builder.patch(body), path, headers, resultMapper);
    }

    /**
     * 执行HTTP DELETE请求
     * @param path 路径
     * @param headers 头部信息
     * @param body 请求体
     * @param resultMapper 结果生成方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    protected <T> @NonNull Response<T> delete(
            @NonNull String path,
            Headers headers,
            RequestBody body,
            @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        return request(builder -> builder.delete(body), path, headers, resultMapper);
    }
}
