package cn.duskykite.open.call;

import cn.duskykite.open.call.requestbody.*;
import cn.duskykite.open.call.result.DataResult;
import cn.duskykite.open.call.result.PostAuthV3AppAccessTokenResult;
import cn.duskykite.open.call.result.PostAuthV3TenantAccessTokenInternalResult;
import cn.duskykite.open.call.result.PostAuthV3TenantAccessTokenResult;
import cn.duskykite.open.call.result.data.*;
import cn.duskykite.open.call.result.mapper.ByteResultMapper;
import com.google.gson.reflect.TypeToken;
import lombok.NonNull;
import okhttp3.*;
import org.checkerframework.checker.nullness.qual.Nullable;

import java.util.Collection;
import java.util.function.Function;

/**
 * 飞书开放接口远程调用对象部分接口实现
 * @author <a href="mailto:wh1zper@qq.com">wh1zper</a>
 */
public class SAOs extends SAO {

    /**
     * @param primaryURL 开放接口主链接，例如<a href="https://open.feishu.cn/open-apis">飞书开放平台接口</a>
     * @param client HTTP请求客户端
     */
    public SAOs(@NonNull HttpUrl primaryURL, @NonNull OkHttpClient client) {
        super(primaryURL, client);
    }

    /**
     * <a href="https://open.feishu.cn/document/ukTMukTMukTM/ukDNz4SO0MjL5QzM/auth-v3/auth/app_access_token">商店应用获取 app_access_token</a>
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<PostAuthV3AppAccessTokenResult> postAuthV3AppAccessToken(
            @NonNull PostAuthV3AppAccessTokenRequestBody body
    ) {
        return post(
                OpenAPIEnum.AUTH_V3_APP_ACCESS_TOKEN, null,
                JSON_CONTENT_TYPE_HEADERS,
                toJsonRequestBody(body),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/ukTMukTMukTM/ukDNz4SO0MjL5QzM/auth-v3/auth/tenant_access_token">商店应用获取 tenant_access_token</a>
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<PostAuthV3TenantAccessTokenResult> postAuthV3TenantAccessToken(
            @NonNull PostAuthV3TenantAccessTokenRequestBody body
    ) {
        return post(
                OpenAPIEnum.AUTH_V3_TENANT_ACCESS_TOKEN, null,
                JSON_CONTENT_TYPE_HEADERS,
                toJsonRequestBody(body),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/ukTMukTMukTM/ukDNz4SO0MjL5QzM/auth-v3/auth/tenant_access_token_internal">自建应用获取 tenant_access_token</a>
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<PostAuthV3TenantAccessTokenInternalResult> postAuthV3TenantAccessTokenInternal(
            @NonNull PostAuthV3TenantAccessTokenInternalRequestBody body
    ) {
        return post(
                OpenAPIEnum.AUTH_V3_TENANT_ACCESS_TOKEN_INTERNAL, null,
                JSON_CONTENT_TYPE_HEADERS,
                toJsonRequestBody(body),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/department/get">获取单个部门信息</a>
     * @param authorization tenant_access_token 或 user_access_token
     * @param departmentId 需要获取的部门ID 不同 ID 的说明及获取方式 <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/department/field-overview#23857fe0">参见部门ID说明</a>
     * @param userIdType 用户 ID 类型 open_id/union_id/user_id 默认值：open_id
     * @param departmentIdType 此次调用中使用的部门ID的类型 不同 ID 的说明 <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/department/field-overview#23857fe0">部门ID说明</a>
     * @return 响应结果
     */
    public @NonNull Response<DataResult<GetContactV3DepartmentsDepartmentIdData>> getContactV3DepartmentsDepartmentId(
            @NonNull String authorization, @NonNull String departmentId, String userIdType, String departmentIdType
    ) {
        return get(
                OpenAPIEnum.CONTACT_V3_DEPARTMENTS_DEPARTMENT_ID, variables(departmentId, userIdType, departmentIdType),
                toAuthorizationHeaders(authorization),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/department/children">获取子部门列表</a>
     * @param authorization tenant_access_token 或 user_access_token
     * @param departmentId 部门ID，根部门的部门ID 为0 department_id的获取方式参见 <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/department/field-overview#23857fe0">部门ID说明</a>
     * @param userIdType 用户 ID 类型 open_id/union_id/user_id 示例值："open_id"
     * @param departmentIdType 此次调用中使用的部门ID的类型 不同 ID 的说明与department_id的获取方式参见 <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/department/field-overview#23857fe0">部门ID说明</a> department_id/open_department_id 默认值：open_department_id
     * @param fetchChild 是否递归获取子部门
     * @param pageSize 分页大小 示例值：10 默认值：10
     * @param pageToken 分页标记，第一次请求不填，表示从头开始遍历；分页查询结果还有更多项时会同时返回新的 page_token，下次遍历可采用该 page_token 获取查询结果
     * @return 响应结果
     */
    public @NonNull Response<DataResult<GetContactV3DepartmentsDepartmentIdChildrenData>>
    getContactV3DepartmentsDepartmentIdChildren(
            @NonNull String authorization,
            @NonNull String departmentId,
            String userIdType,
            String departmentIdType,
            Boolean fetchChild,
            Integer pageSize,
            String pageToken
    ) {
        return get(
                OpenAPIEnum.CONTACT_V3_DEPARTMENTS_DEPARTMENT_ID_CHILDREN,
                variables(departmentId, userIdType, departmentIdType, fetchChild, pageSize, pageToken),
                toAuthorizationHeaders(authorization),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/scope/list">获取通讯录授权范围</a>
     * @param authorization tenant_access_token
     * @param userIdType 用户 ID 类型 open_id/union_id/user_id 默认值：open_id
     * @param departmentIdType 返回值的部门ID的类型 department_id/open_department_id 默认值：open_department_id
     * @param pageToken 分页标记，第一次请求不填，表示从头开始遍历；分页查询结果还有更多项时会同时返回新的 page_token，下次遍历可采用该 page_token 获取查询结果
     * @param pageSize 分页大小 最大值：100
     * @return 响应结果
     */
    public @NonNull Response<DataResult<GetContactV3ScopesData>> getContactV3Scopes(
            @NonNull String authorization,
            String userIdType,
            String departmentIdType,
            String pageToken,
            Integer pageSize
    ) {
        return get(
                OpenAPIEnum.CONTACT_V3_SCOPES, variables(userIdType, departmentIdType, pageToken, pageSize),
                toAuthorizationHeaders(authorization),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/user/find_by_department">获取部门直属用户列表</a>
     * @param authorization tenant_access_token 或 user_access_token
     * @param userIdType 用户 ID 类型 open_id/union_id/user_id 默认值：open_id
     * @param departmentIdType 返回值的部门ID的类型 department_id/open_department_id 默认值：open_department_id
     * @param departmentId 填写该字段表示获取该部门下用户，必填。根部门的部门ID为0。ID值与查询参数中的department_id_type 对应。不同 ID 的说明与department_id的获取方式参见 <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/department/field-overview#23857fe0">部门ID说明</a>
     * @param pageSize 分页大小 默认值：10 最大值：50
     * @param pageToken 分页标记，第一次请求不填，表示从头开始遍历；分页查询结果还有更多项时会同时返回新的 page_token，下次遍历可采用该 page_token 获取查询结果
     * @return 响应结果
     */
    public @NonNull Response<DataResult<GetContactV3UsersFindByDepartmentData>> getContactV3UsersFindByDepartment(
            @NonNull String authorization,
            String userIdType,
            String departmentIdType,
            @NonNull String departmentId,
            Integer pageSize,
            String pageToken
    ) {
        return get(
                OpenAPIEnum.CONTACT_V3_USERS_FIND_BY_DEPARTMENT,
                variables(userIdType, departmentIdType, departmentId, pageSize, pageToken),
                toAuthorizationHeaders(authorization),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/contact-v3/user/get">获取单个用户信息</a>
     * @param authorization tenant_access_token 或 user_access_token
     * @param userId 用户ID，类型需要与查询参数中的user_id_type保持一致。 例如user_id_type=open_id，user_id的类型需为open_id 不同ID的说明参见 <a href="https://open.feishu.cn/document/home/user-identity-introduction/introduction">用户相关的 ID 概念</a>
     * @param userIdType 用户 ID 类型 open_id/union_id/user_id 默认值：open_id
     * @param departmentIdType 返回值的部门ID的类型 department_id/open_department_id 默认值：open_department_id
     * @return 响应结果
     */
    public @NonNull Response<DataResult<GetContactV3UsersUserIdData>> getContactV3UsersUserId(
            @NonNull String authorization, @NonNull String userId, String userIdType, String departmentIdType
    ) {
        return get(
                OpenAPIEnum.CONTACT_V3_USERS_USER_ID, variables(userId, userIdType, departmentIdType),
                toAuthorizationHeaders(authorization),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/image/create">上传图片</a>
     * @param authorization tenant_access_token
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<DataResult<PostIMV1ImageData>> postIMV1Image(
            @NonNull String authorization, @NonNull PostIMV1ImageRequestBody body
    ) {
        var multipartBody = toMultipartBody(body);
        return post(OpenAPIEnum.IM_V1_IMAGE, null,
                headers(toAuthorizationHeaders(authorization), toContentTypeHeaders(multipartBody.contentType())),
                multipartBody,
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/message/create">发送消息</a>
     * @param authorization tenant_access_token
     * @param receiveIdType 消息接收者id类型 open_id/user_id/union_id/email/chat_id
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<DataResult<IMV1Messages>> postIMV1Messages(
            @NonNull String authorization, @NonNull String receiveIdType, @NonNull PostIMV1MessagesRequestBody body
    ) {
        return post(OpenAPIEnum.IM_V1_MESSAGES, variables(receiveIdType),
                headers(toAuthorizationHeaders(authorization), JSON_CONTENT_TYPE_HEADERS),
                toJsonRequestBody(body),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/message/patch">更新应用发送的消息卡片</a>
     * @param authorization tenant_access_token 或 user_access_token
     * @param messageId 待更新的消息的ID，仅支持更新消息卡片（interactive类型），详情参见<a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/message/intro#ac79c1c2">...</a>
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<DataResult<Object>> patchIMV1MessagesMessageId(
            @NonNull String authorization,
            @NonNull String messageId,
            @NonNull PatchIMV1MessagesMessageIdRequestBody body
    ) {
        return patch(
                OpenAPIEnum.IM_V1_MESSAGES_MESSAGE_ID, variables(messageId),
                headers(toAuthorizationHeaders(authorization), JSON_CONTENT_TYPE_HEADERS),
                toJsonRequestBody(body),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/message/reply">回复消息</a>
     * @param authorization tenant_access_token
     * @param messageId 待回复的消息的ID，详情参见 <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/message/intro#ac79c1c2">消息ID说明</a>
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<DataResult<IMV1Messages>> postIMV1MessagesMessageIdReply(
            @NonNull String authorization,
            @NonNull String messageId,
            @NonNull PostIMV1MessagesMessageIdReplyRequestBody body
    ) {
        return post(OpenAPIEnum.IM_V1_MESSAGES_MESSAGE_ID_REPLY, variables(messageId),
                headers(toAuthorizationHeaders(authorization), JSON_CONTENT_TYPE_HEADERS),
                toJsonRequestBody(body),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/message-resource/get">获取消息中的资源文件</a>
     * @param authorization tenant_access_token
     * @param messageId 待查询资源对应的消息ID
     * @param fileKey 待查询资源的key
     * @param type 资源类型 image/file
     * @return 响应结果
     */
    public @NonNull Response<byte[]> getIMV1MessagesResources(
            @NonNull String authorization, @NonNull String messageId, @NonNull String fileKey, @NonNull String type
    ) {
        return get(
                OpenAPIEnum.IM_V1_MESSAGES_RESOURCES, variables(messageId, fileKey, type),
                toAuthorizationHeaders(authorization));
    }

    /**
     * <a href="https://open.feishu.cn/document/ukTMukTMukTM/uMDO1YjLzgTN24yM4UjN">延时更新消息卡片</a>
     * @param authorization tenant_access_token
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<Result> postInteractiveV1CardUpdate(
            @NonNull String authorization, @NonNull PostInteractiveV1CardUpdateRequestBody body
    ) {
        return post(OpenAPIEnum.INTERACTIVE_V1_CARD_UPDATE, null,
                headers(toAuthorizationHeaders(authorization), JSON_CONTENT_TYPE_HEADERS),
                toJsonRequestBody(body),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/ukTMukTMukTM/ucDO1EjL3gTNx4yN4UTM">批量发送消息</a>
     * @param authorization tenant_access_token
     * @param body 请求体
     * @return 响应结果
     */
    public @NonNull Response<DataResult<PostMessageV4BatchSendData>> postMessageV4BatchSend(
            @NonNull String authorization, @NonNull PostMessageV4BatchSendRequestBody body
    ) {
        return post(OpenAPIEnum.MESSAGE_V4_BATCH_SEND, null,
                headers(toAuthorizationHeaders(authorization), JSON_CONTENT_TYPE_HEADERS),
                toJsonRequestBody(body),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/chat/list">获取用户或机器人所在的群列表</a>
     * @param authorization tenant_access_token 或 user_access_token
     * @param userIdType 用户 ID 类型 open_id/union_id/user_id 默认值：open_id
     * @param pageToken 分页标记，第一次请求不填，表示从头开始遍历；分页查询结果还有更多项时会同时返回新的 page_token，下次遍历可采用该 page_token 获取查询结果
     * @param pageSize 分页大小 默认值：20 最大值：100
     * @return 响应结果
     */
    public @NonNull Response<DataResult<GetIMV1ChatsData>> getIMV1Chats(
            @NonNull String authorization, String userIdType, String pageToken, Integer pageSize
    ) {
        return get(OpenAPIEnum.IM_V1_CHATS, variables(userIdType, pageToken, pageSize),
                toAuthorizationHeaders(authorization),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/chat-members/get">获取群成员列表</a>
     * @param authorization tenant_access_token 或 user_access_token
     * @param chatId 群 ID，详情参见 <a href="https://open.feishu.cn/document/uAjLw4CM/ukTMukTMukTM/reference/im-v1/chat-id-description">群ID 说明</a>
     * @param memberIdType 群成员 用户 ID 类型，详情参见 <a href="https://open.feishu.cn/document/home/user-identity-introduction/introduction">用户相关的 ID 概念</a>
     * @param pageToken 分页标记，第一次请求不填，表示从头开始遍历；分页查询结果还有更多项时会同时返回新的 page_token，下次遍历可采用该 page_token 获取查询结果
     * @param pageSize 分页大小 默认值：20 最大值：100
     * @return 响应结果
     */
    public @NonNull Response<DataResult<GetIMV1ChatsMembersData>> getIMV1ChatsMembers(
            @NonNull String authorization,
            @NonNull String chatId,
            String memberIdType,
            String pageToken,
            Integer pageSize
    ) {
        return get(OpenAPIEnum.IM_V1_CHATS_MEMBERS, variables(chatId, memberIdType, pageToken, pageSize),
                toAuthorizationHeaders(authorization),
                toJsonResultMapper(new TypeToken<>(){}));
    }

    /**
     * 根据路径对象与路径中的变量的字符串（多个）生成最终的路径字符串
     * @param openApiEnum 开放接口枚举
     * @param pathVariables 路径中的变量字符串（多个）
     * @return 路径字符串
     */
    private static @NonNull String toPath(
            @NonNull OpenAPIEnum openApiEnum,
            Collection<@Nullable String> pathVariables
    ) {
        return openApiEnum.getPath().resolve(pathVariables);
    }

    /**
     * 执行HTTP GET请求
     * @param openApiEnum 开放接口枚举
     * @param pathVariables 路径中的变量字符串（多个）
     * @param headers 头部信息
     * @param resultMapper 结果生成方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    private <T> @NonNull Response<T> get(
            @NonNull OpenAPIEnum openApiEnum,
            Collection<@org.checkerframework.checker.nullness.qual.Nullable String> pathVariables,
            Headers headers,
            @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        return get(toPath(openApiEnum, pathVariables), headers, resultMapper);
    }

    /**
     * 执行HTTP GET请求
     * @param openApiEnum 开放接口枚举
     * @param pathVariables 路径中的变量字符串（多个）
     * @param headers 头部信息
     * @return 字节数组的响应结果
     */
    private @NonNull Response<byte[]> get(
            @NonNull OpenAPIEnum openApiEnum,
            Collection<@org.checkerframework.checker.nullness.qual.Nullable String> pathVariables,
            Headers headers
    ) {
        return get(openApiEnum, pathVariables, headers, new ByteResultMapper() {});
    }

    /**
     * 执行HTTP POST请求
     * @param openApiEnum 开放接口枚举
     * @param pathVariables 路径中的变量字符串（多个）
     * @param headers 头部信息
     * @param body 请求体
     * @param resultMapper 结果生成方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    private <T> @NonNull Response<T> post(
            @NonNull OpenAPIEnum openApiEnum,
            Collection<@org.checkerframework.checker.nullness.qual.Nullable String> pathVariables,
            Headers headers,
            @NonNull RequestBody body,
            @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        return post(openApiEnum.getPath().resolve(pathVariables), headers, body, resultMapper);
    }

    /**
     * 执行HTTP PATCH请求
     * @param openApiEnum 开放接口枚举
     * @param pathVariables 路径中的变量字符串（多个）
     * @param headers 头部信息
     * @param body 请求体
     * @param resultMapper 结果生成方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    private <T> @NonNull Response<T> patch(
            @NonNull OpenAPIEnum openApiEnum,
            Collection<@org.checkerframework.checker.nullness.qual.Nullable String> pathVariables,
            Headers headers,
            @NonNull RequestBody body,
            @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        return patch(toPath(openApiEnum, pathVariables), headers, body, resultMapper);
    }

    /**
     * 执行HTTP DELETE请求
     * @param openApiEnum 开放接口枚举
     * @param pathVariables 路径中的变量字符串（多个）
     * @param headers 头部信息
     * @param body 请求体
     * @param resultMapper 结果生成方法
     * @return 响应结果
     * @param <T> 响应结果的类型
     */
    private <T> @NonNull Response<T> delete(
            @NonNull OpenAPIEnum openApiEnum,
            Collection<@org.checkerframework.checker.nullness.qual.Nullable String> pathVariables,
            Headers headers,
            RequestBody body,
            @NonNull Function<ResponseBody, ? extends T> resultMapper
    ) {
        return delete(toPath(openApiEnum, pathVariables), headers, body, resultMapper);
    }
}
