package cn.duskykite.open.call.requestbody;

import lombok.Builder;
import lombok.NonNull;

import java.util.List;
import java.util.Objects;

/**
 * 批量发送消息的请求体
 * @author <a href="mailto:wh1zper@qq.com">wh1zper</a>
 */
@Builder
public record PostMessageV4BatchSendRequestBody(
        // 消息类型，支持多种消息类型，详见消息类型及内容示例 https://open.feishu.cn/document/ukTMukTMukTM/ucDO1EjL3gTNx4yN4UTM#42562565
        @NonNull String msgType,
        // 消息内容，支持除卡片消息外的多种消息内容，详见消息类型及内容示例 https://open.feishu.cn/document/ukTMukTMukTM/ucDO1EjL3gTNx4yN4UTM#42562565
        Content content,
        // 卡片消息内容
        // 注意：card和content字段必须二选一
        Card card,
        // 支持自定义部门ID https://open.feishu.cn/document/ukTMukTMukTM/uYTM5UjL2ETO14iNxkTN/terminology#3c3e6267 和open_department_id，列表长度小于等于 200
        // 注意：部门下的所有子部门包含的成员也会收到消息
        List<String> departmentIds,
        // 用户 open_id 列表，长度小于等于 200；ID获取方式可参考文档如何获取 Open ID？ https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-openid
        List<String> openIds,
        // 用户 user_id 列表，长度小于等于 200；ID获取方式可参考文档如何获取 User ID？ https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-user-id （对应 V3 接口的 employee_ids ）
        List<String> userIds,
        // 用户 union_ids 列表，长度小于等于 200；ID获取方式可参考文档如何获取 Union ID？ https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-union-id
        List<String> unionIds
) {
    /**
     * constructor
     * @param msgType 消息类型，支持多种消息类型，详见 <a href="https://open.feishu.cn/document/ukTMukTMukTM/ucDO1EjL3gTNx4yN4UTM#42562565">消息类型及内容示例</a>
     * @param content 消息内容，支持除卡片消息外的多种消息内容，详见 <a href="https://open.feishu.cn/document/ukTMukTMukTM/ucDO1EjL3gTNx4yN4UTM#42562565">消息类型及内容示例</a>
     * @param card 卡片消息内容 注意：card和content字段必须二选一
     * @param departmentIds 支持<a href="https://open.feishu.cn/document/ukTMukTMukTM/uYTM5UjL2ETO14iNxkTN/terminology#3c3e6267">自定义部门ID</a>和open_department_id，列表长度小于等于 200 注意：部门下的所有子部门包含的成员也会收到消息
     * @param openIds 用户 open_id 列表，长度小于等于 200；ID获取方式可参考文档 <a href="https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-openid">如何获取 Open ID？</a>
     * @param userIds 用户 user_id 列表，长度小于等于 200；ID获取方式可参考文档 <a href="https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-user-id">如何获取 User ID？</a> （对应 V3 接口的 employee_ids ）
     * @param unionIds 用户 union_ids 列表，长度小于等于 200；ID获取方式可参考文档 <a href="https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-union-id">如何获取 Union ID？</a>
     */
    public PostMessageV4BatchSendRequestBody {
        if (Objects.isNull(content) && Objects.isNull(card)) {
            throw new NullPointerException();
        }

        if (Objects.isNull(departmentIds) &&
                Objects.isNull(openIds) &&
                Objects.isNull(userIds) &&
                Objects.isNull(unionIds)) {
            throw new NullPointerException();
        }
    }
}
