package cn.duskykite.open.call.result.data;

import cn.duskykite.open.Xs;
import lombok.Getter;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;

import java.util.Optional;

/**
 * 用户 ID 类型枚举类
 * @author <a href="mailto:wh1zper@qq.com">wh1zper</a>
 */
@RequiredArgsConstructor
@Getter
public enum UserIdTypeEnum {
    /**
     * 标识一个用户在某个应用中的身份。同一个用户在不同应用中的 Open ID 不同。<a href="https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-openid">了解更多：如何获取 Open ID</a>
     */
    OPEN_ID("open_id"),
    /**
     * 标识一个用户在某个应用开发商下的身份。同一用户在同一开发商下的应用中的 Union ID 是相同的，在不同开发商下的应用中的 Union ID 是不同的。通过 Union ID，应用开发商可以把同个用户在多个应用中的身份关联起来。<a href="https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-union-id">了解更多：如何获取 Union ID？</a>
     */
    UNION_ID("union_id"),
    /**
     * 标识一个用户在某个租户内的身份。同一个用户在租户 A 和租户 B 内的 User ID 是不同的。在同一个租户内，一个用户的 User ID 在所有应用（包括商店应用）中都保持一致。User ID 主要用于在不同的应用间打通用户数据。<a href="https://open.feishu.cn/document/uAjLw4CM/ugTN1YjL4UTN24CO1UjN/trouble-shooting/how-to-obtain-user-id">了解更多：如何获取 User ID？</a>
     */
    USER_ID("user_id"),
    ;

    /**
     * 根据类型值获取枚举
     * @param type 类型值
     * @return 枚举
     */
    public static UserIdTypeEnum getByType(String type) {
        return Xs.findByIdentifierIfNonnull(type, values(), e -> e.type);
    }

    /**
     * 类型值
     */
    @NonNull
    private final String type;
}
