package cn.fyupeng.protocol;

import cn.fyupeng.enums.ResponseCode;
import cn.fyupeng.exception.ReceiveResponseException;
import cn.fyupeng.exception.RpcException;
import cn.fyupeng.serializer.CommonSerializer;
import cn.fyupeng.util.AesEncoder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.Objects;

/**
 * @Auther: fyp
 * @Date: 2022/3/28
 * @Description:
 * @Package: cn.fyupeng.util
 * @Version: 1.0
 */
@Slf4j
public class RpcMessageChecker {

    private static final CommonSerializer serializer = CommonSerializer.getByCode(CommonSerializer.JSON_SERIALIZER);

    public RpcMessageChecker() {
    }

    /**
     * true 通过校验
     * false 校验失败
     * @param rpcRequest rpc 待校验 请求包
     * @param rpcResponse rpc 待校验 响应包
     * @return boolean
     * @throws RpcException
     */
    public static boolean check(RpcRequest rpcRequest, RpcResponse rpcResponse) throws RpcException {
        if (rpcResponse == null) {
            log.error("service call failed, service: {}",rpcRequest.getInterfaceName());
            return false;
            //throw new ReceiveResponseException("service call failed Exception");
        }
        /**
         *  校验请求包 请求号 与 响应包中的 请求号 是否一致
         */
        if (!rpcResponse.getRequestId().equals(rpcRequest.getRequestId())) {
            log.error("inconsistent request numbers");
            return false;
            //throw new ReceiveResponseException("inconsistent request numbers Exception");
        }

        /**
         * 注意 rpcResponse 是 通过 反序列化 重新 实例化的 对象
         * rpcResponse.getStatusCode() 与 ResponseCode.SUCCESS.getCode() 不是同一个对象，虽然 ResponseCode 是单例模式
         * equals() 方法 判断 的 是 两个对象 的 值 相等 切忌 使用 !=
         */
        if (rpcResponse.getStatusCode() == null || !rpcResponse.getStatusCode().equals(ResponseCode.SUCCESS.getCode())) {
            log.error("service call failed, service: {}",rpcRequest.getInterfaceName());
            return false;
            //throw new ReceiveResponseException("service call failed Exception");
        }

        /**
         * 校验包 是否被人 修改过
         */
        String checkCode;
        // data 为空 校验码为 null
        if(rpcResponse.getData() == null) {
            checkCode = null;
            // data 有值 设置 校验码
        } else {
            byte[] checkData = serializer.serialize(rpcResponse.getData());
            checkCode = AesEncoder.encrypt(new String(checkData));
        }
        // data 应为 null
        if(rpcResponse.getCheckCode() == null) {
            // 服务端校验码 与 客户端校验码 不一致
            // checkCode 由 data 计算而来，发送前 校验码为 null，此时不一致，说明 data 数据被修改
            if(!Objects.equals(checkCode, rpcResponse.getCheckCode())) {
                log.error("data in package is modified， data: {}", rpcResponse.getData());
                log.error("detail modification information: {}，the modification information has been filtered, and such messages will not be received and consumed！", rpcResponse.getData().toString());
                return false;
                //throw new ReceiveResponseException("data in package is modified Exception");
            }
            // data 有值
            // 有 返回值的 情况
        } else {
            // 计算两者的 校验码，不一致则 说明 data 数据被 修改
            if (!StringUtils.equals(removeSpecialChars(checkCode), removeSpecialChars(rpcResponse.getCheckCode()))){
                log.error("data in package is modified， data:{}",rpcResponse.getData());
                log.error("detail modification information: {}，the modification information has been filtered, and such messages will not be received and consumed！", rpcResponse.getData().toString());
                return false;
                //throw new ReceiveResponseException("data in package is modified Exception");
            }
        }

        log.debug("Packet verification succeeded!");
        return true;
    }

    /**
     * 校验失败直接抛出异常，外层逻辑则会将该包抛弃不处理
     * @param rpcRequest rpc 校验 请求包
     * @param rpcResponse rpc 校验 响应包
     * @throws RpcException
     */
    public static void checkAndThrow(RpcRequest rpcRequest, RpcResponse rpcResponse) throws RpcException {
        if (rpcResponse == null) {
            log.error("service call failed, service: {}",rpcRequest.getInterfaceName());
            throw new ReceiveResponseException("service call failed Exception");
        }
        /**
         *  校验请求包 请求号 与 响应包中的 请求号 是否一致
         */
        if (!rpcResponse.getRequestId().equals(rpcRequest.getRequestId())) {
            log.error("inconsistent request numbers");
            throw new ReceiveResponseException("inconsistent request numbers Exception");
        }

        /**
         * 注意 rpcResponse 是 通过 反序列化 重新 实例化的 对象
         * rpcResponse.getStatusCode() 与 ResponseCode.SUCCESS.getCode() 不是同一个对象，虽然 ResponseCode 是单例模式
         * equals() 方法 判断 的 是 两个对象 的 值 相等 切忌 使用 !=
         */
        if (rpcResponse.getStatusCode() == null || !rpcResponse.getStatusCode().equals(ResponseCode.SUCCESS.getCode())) {
            log.error("service call failed, service: {}",rpcRequest.getInterfaceName());
            throw new ReceiveResponseException("service call failed Exception");
        }

        /**
         * 校验包 是否被人 修改过
         */
        String checkCode;
        // data 为空 校验码为 null
        if(rpcResponse.getData() == null) {
            checkCode = null;
        // data 有值 设置 校验码
        } else {
            byte[] checkData = serializer.serialize(rpcResponse.getData());
            checkCode = AesEncoder.encrypt(new String(checkData));
        }
        // data 应为 null
        if(rpcResponse.getCheckCode() == null) {
            // 服务端校验码 与 客户端校验码 不一致
            // checkCode 由 data 计算而来，发送前 校验码为 null，此时不一致，说明 data 数据被修改
            if(!Objects.equals(checkCode, rpcResponse.getCheckCode())) {
                log.error("data in package is modified， data: {}", rpcResponse.getData());
                log.error("detail modification information: {}，the modification information has been filtered, and such messages will not be received and consumed！", rpcResponse.getData().toString());
                throw new ReceiveResponseException("data in package is modified Exception");
            }
        // data 有值
        // 有 返回值的 情况
        } else {
            // 计算两者的 校验码，不一致则 说明 data 数据被 修改
            if(!StringUtils.equals(removeSpecialChars(checkCode), removeSpecialChars(rpcResponse.getCheckCode()))) {
                log.error("data in package is modified， data:{}",rpcResponse.getData());
                log.error("detail modification information: {}，the modification information has been filtered, and such messages will not be received and consumed！", rpcResponse.getData().toString());
                throw new ReceiveResponseException("data in package is modified Exception");
            }
        }

        log.debug("Packet verification succeeded!");
    }

    private static String removeSpecialChars(String input) {
        // 使用 StringUtils.replaceChars 方法去除指定的特殊字符
        return StringUtils.replaceChars(input, "_-+/", "");
    }

}
