package cn.fyupeng.net.netty.server;

import cn.fyupeng.handler.RequestHandler;
import cn.fyupeng.protocol.RpcRequest;
import cn.fyupeng.protocol.RpcResponse;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelHandlerContext;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.DefaultEventExecutorGroup;
import io.netty.util.concurrent.EventExecutorGroup;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.HashSet;

/**
 * @Auther: fyp
 * @Date: 2023/1/7
 * @Description: Netty Channel分发器
 * @Package: cn.fyupeng.net.netty.server
 * @Version: 1.0
 */
@Slf4j
public class NettyChannelDispatcher {

    /**
     * netty 服务端采用 线程池处理耗时任务
     */
    private static final EventExecutorGroup group = new DefaultEventExecutorGroup(16);

    private static RequestHandler requestHandler;
    /**
     * 超时重试请求id 集
     */
    private static HashSet<String> timeoutRetryRequestIdSet = new HashSet<>();

    /**
     * 保存上一次的请求执行 结果
     */
    // 多线程可超时幂等性处理
    private static HashMap<String, Object> resMap = new HashMap<>();

    static {
        requestHandler = new RequestHandler();
    }

    public static void init() {
        log.info("netty channel dispatcher initialize successfully!");
    }

    public static void dispatch(ChannelHandlerContext ctx, RpcRequest msg) {
        group.submit(new Runnable() {
            @Override
            public void run() {
                try {
                    log.info("server has received request package: {}", msg);
                    // 到了这一步，如果请求包在上一次已经被 服务器成功执行，接下来要做幂等性处理，也就是客户端设置超时重试处理
                    /**
                     * 这里要防止重试
                     * 分为两种情况
                     * 1. 如果是 客户端发送给服务端 途中出现问题，请求包之前 服务器未获取到，也就是 唯一请求id号 没有重复
                     * 2. 如果是 服务端发回客户端途中出现问题，导致客户端触发 超时重试，这时服务端会 接收 重试请求包，也就是有 重复请求id号
                     */
                    // 请求id 为第一次请求 id
                    Object result = null;
                    if (timeoutRetryRequestIdSet.add(msg.getRequestId())) {
                        result = requestHandler.handler(msg);
                        resMap.put(msg.getRequestId(), result);
                        //请求id 为第二次或以上请求
                    } else {
                        result = resMap.get(msg.getRequestId());
                    }
                    // 生成 校验码，客户端收到后 会 对 数据包 进行校验
                    if (ctx.channel().isActive() && ctx.channel().isWritable()) {
                        /**
                         * 这里要分两种情况：
                         * 1. 当数据无返回值时，保证 checkCode 与 result 可以检验，客户端 也要判断 result 为 null 时 checkCode 是否也为 null，才能认为非他人修改
                         * 2. 当数据有返回值时，校验 checkCode 与 result 的 md5 码 是否相同
                         */
                        String checkCode = "";
                        // 这里做了 当 data为 null checkCode 为 null，checkCode可作为 客户端的判断 返回值 依据
                        if(result != null) {
                            try {
                                checkCode = new String(DigestUtils.md5(result.toString().getBytes("UTF-8")));
                            } catch (UnsupportedEncodingException e) {
                                log.error("binary stream conversion failure: ", e);
                                //e.printStackTrace();
                            }
                        } else {
                            checkCode = null;
                        }
                        RpcResponse rpcResponse = RpcResponse.success(result, msg.getRequestId(), checkCode);
                        log.info(String.format("server send back response package {requestId: %s, message: %s, statusCode: %s ]}", rpcResponse.getRequestId(), rpcResponse.getMessage(), rpcResponse.getStatusCode()));
                        ChannelFuture future = ctx.writeAndFlush(rpcResponse);

                        /**
                         * 大于 1000 条请求id 时，及时清除不用的请求 id
                         * 保存此时 服务接收的请求 id
                         * 考虑多线程中 对其他 线程刚添加的请求id 进行清除的影响
                         */
                        if (timeoutRetryRequestIdSet.size() >= 1000) {
                            synchronized (this) {
                                if (timeoutRetryRequestIdSet.size() >= 1000) {
                                    timeoutRetryRequestIdSet.clear();
                                    resMap.clear();
                                    timeoutRetryRequestIdSet.add(msg.getRequestId());
                                    resMap.put(msg.getRequestId(), result);
                                }
                            }
                        }
                    } else {
                        log.error("channel is not writable");
                    }
                    //}

                    /**
                     * 1. 通道关闭后，对于 心跳包 将不可用
                     * 2. 由于客户端 使用了 ChannelProvider 来 缓存 channel，这里关闭后，无法 发挥 channel 缓存的作用
                     */
                    //future.addListener(ChannelFutureListener.CLOSE);
                } finally {
                    ReferenceCountUtil.release(msg);
                }
            }
        });
    }

}
