package cn.handyplus.lib.attribute;

import cn.handyplus.lib.constants.BaseConstants;
import cn.handyplus.lib.constants.VersionCheckEnum;
import cn.handyplus.lib.core.CollUtil;
import cn.handyplus.lib.expand.adapter.PlayerSchedulerUtil;
import com.google.gson.Gson;
import org.bukkit.entity.Player;
import org.bukkit.potion.PotionEffect;
import org.bukkit.potion.PotionEffectType;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

/**
 * 药水效果
 *
 * @author handy
 */
public class PotionEffectUtil {

    protected static final Map<UUID, List<PotionEffectType>> PLAYER_EFFECT_MAP = new ConcurrentHashMap<>();

    private PotionEffectUtil() {
    }

    /**
     * 给玩家添加属性
     *
     * @param player        玩家
     * @param attributeList 属性
     */
    protected static void addAttribute(Player player, List<String> attributeList) {
        // 转化格式
        List<PotionEffectParam> peList = new ArrayList<>();
        for (String attribute : attributeList) {
            PotionEffectParam potionEffectParam = new Gson().fromJson(attribute, PotionEffectParam.class);
            peList.add(potionEffectParam);
        }
        // 处理药水添加
        List<PotionEffect> potionEffects = new ArrayList<>();
        List<PotionEffectType> potionEffectTypeList = new ArrayList<>();
        for (PotionEffectParam potionEffectParam : peList) {
            PotionEffectType potionEffectType = PotionEffectType.getByName(potionEffectParam.getPotionName());
            if (potionEffectType == null) {
                continue;
            }
            PotionEffect potionEffect = getPotionEffect(potionEffectParam, potionEffectType);
            potionEffects.add(potionEffect);
            potionEffectTypeList.add(potionEffectType);
        }
        if (!potionEffects.isEmpty()) {
            PlayerSchedulerUtil.addPotionEffects(player, potionEffects);
            // 缓存数据
            List<PotionEffectType> typeList = PLAYER_EFFECT_MAP.getOrDefault(player.getUniqueId(), new ArrayList<>());
            typeList.addAll(potionEffectTypeList);
            typeList = typeList.stream().distinct().collect(Collectors.toList());
            PLAYER_EFFECT_MAP.put(player.getUniqueId(), typeList);
        }
    }

    /**
     * 构建PotionEffect
     *
     * @param potionEffectParam 药水参数
     * @param potionEffectType  药水类型
     * @return PotionEffect
     */
    private static PotionEffect getPotionEffect(PotionEffectParam potionEffectParam, PotionEffectType potionEffectType) {
        int potionLevel = potionEffectParam.getPotionLevel() - 1;
        if (potionLevel < 0) {
            potionLevel = 0;
        }
        PotionEffect potionEffect;
        // 默认一天
        if (BaseConstants.VERSION_ID < VersionCheckEnum.V_1_8.getVersionId()) {
            potionEffect = new PotionEffect(potionEffectType, 1728000, potionLevel, false);
        } else {
            potionEffect = new PotionEffect(potionEffectType, 1728000, potionLevel, false, !potionEffectParam.getPotionHide());
        }
        return potionEffect;
    }

    /**
     * 移除玩家药水属性
     *
     * @param player 玩家
     */
    protected static void removeAttribute(Player player) {
        Collection<PotionEffect> activePotionEffects = player.getActivePotionEffects();
        if (activePotionEffects.isEmpty()) {
            return;
        }
        List<PotionEffectType> potionEffectTypeList = PLAYER_EFFECT_MAP.get(player.getUniqueId());
        if (CollUtil.isEmpty(potionEffectTypeList)) {
            return;
        }
        for (PotionEffect potionEffect : activePotionEffects) {
            if (potionEffectTypeList.contains(potionEffect.getType())) {
                PlayerSchedulerUtil.removePotionEffect(player, potionEffect.getType());
            }
        }
        PLAYER_EFFECT_MAP.remove(player.getUniqueId());
    }

}