/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.core.definition.support;

import cn.herodotus.stirrup.core.definition.constants.ErrorCodes;
import cn.herodotus.stirrup.core.definition.domain.ErrorCodeMapper;
import cn.herodotus.stirrup.core.definition.domain.Feedback;
import cn.herodotus.stirrup.core.definition.feedback.*;
import org.apache.commons.collections4.MapUtils;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * <p>Description: 错误码错误码构建器 </p>
 * <p>
 * 1**	信息，服务器收到请求，需要请求者继续执行操作
 * 2**	成功，操作被成功接收并处理
 * 3**	重定向，需要进一步的操作以完成请求
 * 4**	客户端错误，请求包含语法错误或无法完成请求
 * 5**	服务器错误，服务器在处理请求的过程中发生了错误
 * <p>
 * 1开头的状态码
 * 100	Continue	继续。客户端应继续其请求
 * 101	Switching Protocols	切换协议。服务器根据客户端的请求切换协议。只能切换到更高级的协议，例如，切换到HTTP的新版本协议
 * <p>
 * 2开头的状态码
 * 200	OK	请求成功。一般用于GET与POST请求
 * 201	Created	已创建。成功请求并创建了新的资源
 * 202	Accepted	已接受。已经接受请求，但未处理完成
 * 203	Non-Authoritative Information	非授权信息。请求成功。但返回的meta信息不在原始的服务器，而是一个副本
 * 204	No Content	无内容。服务器成功处理，但未返回内容。在未更新网页的情况下，可确保浏览器继续显示当前文档
 * 205	Reset Content	重置内容。服务器处理成功，用户终端（例如：浏览器）应重置文档视图。可通过此返回码清除浏览器的表单域
 * 206	Partial Content	部分内容。服务器成功处理了部分GET请求
 * <p>
 * 3开头的状态码
 * 300	Multiple Choices	多种选择。请求的资源可包括多个位置，相应可返回一个资源特征与地址的列表用于用户终端（例如：浏览器）选择
 * 301	Moved Permanently	永久移动。请求的资源已被永久的移动到新URI，返回信息会包括新的URI，浏览器会自动定向到新URI。今后任何新的请求都应使用新的URI代替
 * 302	Found	临时移动。与301类似。但资源只是临时被移动。客户端应继续使用原有URI
 * 303	See Other	查看其它地址。与301类似。使用GET和POST请求查看
 * 304	Not Modified	未修改。所请求的资源未修改，服务器返回此状态码时，不会返回任何资源。客户端通常会缓存访问过的资源，通过提供一个头信息指出客户端希望只返回在指定日期之后修改的资源
 * 305	Use Proxy	使用代理。所请求的资源必须通过代理访问
 * 306	Unused	已经被废弃的HTTP状态码
 * 307	Temporary Redirect	临时重定向。与302类似。使用GET请求重定向
 * <p>
 * 4开头的状态码
 * 400	Bad Request	客户端请求的语法错误，服务器无法理解
 * 401	Unauthorized	请求要求用户的身份认证
 * 402	Payment Required	保留，将来使用
 * 403	Forbidden	服务器理解请求客户端的请求，但是拒绝执行此请求
 * 404	Not Found	服务器无法根据客户端的请求找到资源（网页）。通过此代码，网站设计人员可设置"您所请求的资源无法找到"的个性页面
 * 405	Method Not Allowed	客户端请求中的方法被禁止
 * 406	Not Acceptable	服务器无法根据客户端请求的内容特性完成请求
 * 407	Proxy Authentication Required	请求要求代理的身份认证，与401类似，但请求者应当使用代理进行授权
 * 408	Request Time-out	服务器等待客户端发送的请求时间过长，超时
 * 409	Conflict	服务器完成客户端的PUT请求是可能返回此代码，服务器处理请求时发生了冲突
 * 410	Gone	客户端请求的资源已经不存在。410不同于404，如果资源以前有现在被永久删除了可使用410代码，网站设计人员可通过301代码指定资源的新位置
 * 411	Length Required	服务器无法处理客户端发送的不带Content-Length的请求信息
 * 412	Precondition Failed	客户端请求信息的先决条件错误
 * 413	Request Entity Too Large	由于请求的实体过大，服务器无法处理，因此拒绝请求。为防止客户端的连续请求，服务器可能会关闭连接。如果只是服务器暂时无法处理，则会包含一个Retry-After的响应信息
 * 414	Request-URI Too Large	请求的URI过长（URI通常为网址），服务器无法处理
 * 415	Unsupported Media Type	服务器无法处理请求附带的媒体格式
 * 416	Requested range not satisfiable	客户端请求的范围无效
 * 417	Expectation Failed	服务器无法满足Expect的请求头信息
 * <p>
 * 5开头的状态码
 * 500	Internal Server Error	服务器内部错误，无法完成请求
 * 501	Not Implemented	服务器不支持请求的功能，无法完成请求
 * 502	Bad Gateway	充当网关或代理的服务器，从远端服务器接收到了一个无效的请求
 * 503	Service Unavailable	由于超载或系统维护，服务器暂时的无法处理客户端的请求。延时的长度可包含在服务器的Retry-After头信息中
 * 504	Gateway Time-out	充当网关或代理的服务器，未及时从远端服务器获取请求
 * 505	HTTP Version not supported	服务器不支持请求的HTTP协议的版本，无法完成处理
 *
 * @author : gengwei.zheng
 * @date : 2023/9/24 23:04
 */
public class ErrorCodeMapperBuilder {

    /**
     * 401	Unauthorized	请求要求用户的身份认证
     */
    private final Map<Feedback, Integer> unauthorizedConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.UNAUTHORIZED, ErrorCodes.UNAUTHORIZED.getSequence());
    }};
    /**
     * 403	Forbidden	服务器理解请求客户端的请求，但是拒绝执行此请求
     */
    private final Map<Feedback, Integer> forbiddenConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.FORBIDDEN, ErrorCodes.FORBIDDEN.getSequence());
    }};
    /**
     * 405	Method Not Allowed	客户端请求中的方法被禁止
     */
    private final Map<Feedback, Integer> methodNotAllowedConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.METHOD_NOT_ALLOWED, ErrorCodes.METHOD_NOT_ALLOWED.getSequence());
    }};
    /**
     * 406	Not Acceptable	服务器无法根据客户端请求的内容特性完成请求
     */
    private final Map<Feedback, Integer> notAcceptableConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.NOT_ACCEPTABLE, ErrorCodes.NOT_ACCEPTABLE.getSequence());
    }};
    /**
     * 412	Precondition Failed	客户端请求信息的先决条件错误
     */
    private final Map<Feedback, Integer> preconditionFailedConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.PRECONDITION_FAILED, ErrorCodes.PRECONDITION_FAILED.getSequence());
    }};
    /**
     * 415	Unsupported Media Type	服务器无法处理请求附带的媒体格式
     */
    private final Map<Feedback, Integer> unsupportedMediaTypeConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.PRECONDITION_FAILED, ErrorCodes.PRECONDITION_FAILED.getSequence());
    }};
    /**
     * 500	Internal Server Error	服务器内部错误，无法完成请求
     */
    private final Map<Feedback, Integer> internalServerErrorConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.INTERNAL_SERVER_ERROR, ErrorCodes.INTERNAL_SERVER_ERROR.getSequence());
    }};
    /**
     * 501	Not Implemented	服务器不支持请求的功能，无法完成请求
     */
    private final Map<Feedback, Integer> notImplementedConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.NOT_IMPLEMENTED, ErrorCodes.NOT_IMPLEMENTED.getSequence());
    }};
    /**
     * 503	Service Unavailable	由于超载或系统维护，服务器暂时的无法处理客户端的请求。延时的长度可包含在服务器的Retry-After头信息中
     */
    private final Map<Feedback, Integer> serviceUnavailableConfigs = new LinkedHashMap<>() {{
        put(ErrorCodes.SERVICE_UNAVAILABLE, ErrorCodes.SERVICE_UNAVAILABLE.getSequence());
        put(ErrorCodes.OPEN_API_REQUEST_FAILURE, ErrorCodes.OPEN_API_REQUEST_FAILURE.getSequence());
    }};

    private final Map<Integer, Map<Feedback, Integer>> customizeConfigs = new LinkedHashMap<>();

    private ErrorCodeMapperBuilder create(Map<Feedback, Integer> container, Feedback... items) {
        for (Feedback item : items) {
            container.put(item, item.getSequence(container.size()));
        }
        return this;
    }

    public ErrorCodeMapperBuilder customize(CustomizeFeedback... items) {
        for (Feedback item : items) {
            if (item.isCustom()) {
                Map<Feedback, Integer> config = customizeConfigs.get(item.getCustom());
                if (MapUtils.isEmpty(config)) {
                    config = new LinkedHashMap<>();
                }
                config.put(item, item.getSequence(config.size()));
                customizeConfigs.put(item.getCustom(), config);
            }
        }
        return this;
    }

    /**
     * 401	Unauthorized	请求要求用户的身份认证
     *
     * @param items 错误标识条目  {@link UnauthorizedFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder unauthorized(UnauthorizedFeedback... items) {
        return create(this.unauthorizedConfigs, items);
    }

    /**
     * 403	Forbidden	服务器理解请求客户端的请求，但是拒绝执行此请求
     *
     * @param items 错误标识条目 {@link ForbiddenFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder forbidden(ForbiddenFeedback... items) {
        return create(this.forbiddenConfigs, items);
    }

    /**
     * 405	Method Not Allowed	客户端请求中的方法被禁止
     *
     * @param items 错误标识条目 {@link MethodNotAllowedFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder methodNotAllowed(MethodNotAllowedFeedback... items) {
        return create(this.methodNotAllowedConfigs, items);
    }

    /**
     * 406	Not Acceptable	服务器无法根据客户端请求的内容特性完成请求
     *
     * @param items 错误标识条目 {@link NotAcceptableFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder notAcceptable(NotAcceptableFeedback... items) {
        return create(this.notAcceptableConfigs, items);
    }

    /**
     * 412	Precondition Failed	客户端请求信息的先决条件错误
     *
     * @param items 错误标识条目 {@link PreconditionFailedFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder preconditionFailed(PreconditionFailedFeedback... items) {
        return create(this.preconditionFailedConfigs, items);
    }

    /**
     * 415	Unsupported Media Type	服务器无法处理请求附带的媒体格式
     *
     * @param items 错误标识条目 {@link UnsupportedMediaTypeFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder unsupportedMediaType(UnsupportedMediaTypeFeedback... items) {
        return create(this.unsupportedMediaTypeConfigs, items);
    }

    /**
     * 500	Internal Server Error	服务器内部错误，无法完成请求
     *
     * @param items 错误标识条目 {@link InternalServerErrorFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder internalServerError(InternalServerErrorFeedback... items) {
        return create(this.internalServerErrorConfigs, items);
    }

    /**
     * 501	Not Implemented	服务器不支持请求的功能，无法完成请求
     *
     * @param items 错误标识条目 {@link NotImplementedFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder notImplemented(NotImplementedFeedback... items) {
        return create(this.notImplementedConfigs, items);
    }

    /**
     * 503	Service Unavailable	由于超载或系统维护，服务器暂时的无法处理客户端的请求。延时的长度可包含在服务器的Retry-After头信息中
     *
     * @param items 错误标识条目 {@link ServiceUnavailableFeedback}
     * @return {@link ErrorCodeMapperBuilder}
     */
    public ErrorCodeMapperBuilder serviceUnavailable(ServiceUnavailableFeedback... items) {
        return create(this.serviceUnavailableConfigs, items);
    }

    public ErrorCodeMapper build() {
        ErrorCodeMapper errorCodeMapper = ErrorCodeMapper.getInstance();
        errorCodeMapper.append(unauthorizedConfigs);
        errorCodeMapper.append(forbiddenConfigs);
        errorCodeMapper.append(methodNotAllowedConfigs);
        errorCodeMapper.append(notAcceptableConfigs);
        errorCodeMapper.append(preconditionFailedConfigs);
        errorCodeMapper.append(unsupportedMediaTypeConfigs);
        errorCodeMapper.append(internalServerErrorConfigs);
        errorCodeMapper.append(notImplementedConfigs);
        errorCodeMapper.append(serviceUnavailableConfigs);

        customizeConfigs.forEach((key, feedbacks) -> errorCodeMapper.append(feedbacks));
        return errorCodeMapper;
    }
}
