/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.core.identity.domain;

import cn.herodotus.stirrup.core.identity.enums.PermissionExpression;
import cn.herodotus.stirrup.core.identity.exception.SecurityAttributeClassNotFoundException;
import com.google.common.base.MoreObjects;
import com.google.common.base.Objects;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.dromara.hutool.core.reflect.TypeUtil;
import org.dromara.hutool.core.reflect.method.MethodUtil;
import org.dromara.hutool.extra.spring.SpringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.util.MethodInvocationUtils;

import java.io.Serializable;
import java.lang.reflect.Method;

/**
 * <p>Description: 自定义SecurityConfig </p>
 * <p>
 * 自定义SecurityConfig，主要为了构建无参数构造函数，以解决序列化出错问题
 *
 * @author : gengwei.zheng
 * @date : 2021/9/11 15:57
 */
public class HerodotusSecurityAttribute implements Serializable {

    private static final Logger log = LoggerFactory.getLogger(HerodotusSecurityAttribute.class);

    private String expression;

    private String className;

    private String methodName;

    public HerodotusSecurityAttribute() {
    }

    public String getExpression() {
        return expression;
    }

    public void setExpression(String expression) {
        this.expression = expression;
    }

    public String getClassName() {
        return className;
    }

    public void setClassName(String className) {
        this.className = className;
    }

    public String getMethodName() {
        return methodName;
    }

    public void setMethodName(String methodName) {
        this.methodName = methodName;
    }

    public static HerodotusSecurityAttribute create(String expression, String className, String methodName) {
        HerodotusSecurityAttribute attribute = new HerodotusSecurityAttribute();
        attribute.setExpression(expression);
        attribute.setClassName(className);
        attribute.setMethodName(methodName);
        return attribute;
    }

    public static HerodotusSecurityAttribute createDefaultPermitAll() {
        HerodotusSecurityAttribute attribute = new HerodotusSecurityAttribute();
        attribute.setExpression(PermissionExpression.PERMIT_ALL.getValue());
        attribute.setClassName(HerodotusSecurityAttribute.class.getName());
        attribute.setMethodName("getExpression");
        return attribute;
    }

    public static MethodInvocation createMethodInvocation(HerodotusSecurityAttribute attribute) {
        try {
            Class<?> clazz = ClassUtils.getClass(attribute.getClassName());
            Object object = SpringUtil.getBean(clazz);

            if (ObjectUtils.isEmpty(object)) {
                return MethodInvocationUtils.createFromClass(new HerodotusSecurityAttribute(), HerodotusSecurityAttribute.class, attribute.getMethodName(), null, null);
            } else {
                Method method = MethodUtil.getMethodByName(clazz, attribute.getMethodName());
                Class<?>[] classArgs = TypeUtil.getParamClasses(method);
                return MethodInvocationUtils.createFromClass(object, clazz, attribute.getMethodName(), classArgs, null);
            }
        } catch (ClassNotFoundException e) {
            log.error("[Herodotus] |- Reactive createMethodInvocation error, can not found the class [{}]", attribute.getClassName());
            throw new SecurityAttributeClassNotFoundException(e);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        HerodotusSecurityAttribute that = (HerodotusSecurityAttribute) o;
        return Objects.equal(expression, that.expression);
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(expression);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("attrib", expression)
                .toString();
    }
}
