/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.web.core.reactive.utils;

import cn.herodotus.stirrup.web.core.definition.utils.HttpUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.http.HttpCookie;
import org.springframework.http.HttpHeaders;
import org.springframework.util.MultiValueMap;
import org.springframework.web.server.ServerWebExchange;

/**
 * <p>Description: WebFlux Cookie 工具类 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/2/5 21:11
 */
public class CookieUtils extends HttpUtils {

    /**
     * 读取 {@link ServerWebExchange} 中的 {@link HttpCookie}
     *
     * @param exchange {@link ServerWebExchange}
     * @return MultiValueMap<String, HttpCookie>
     */
    public static MultiValueMap<String, HttpCookie> getCookies(ServerWebExchange exchange) {
        return exchange.getRequest().getCookies();
    }

    /**
     * 读取 {@link ServerWebExchange} 中的某个名称对应的 {@link HttpCookie}
     *
     * @param exchange {@link ServerWebExchange}
     * @param name     Cookie Key
     * @return List<HttpCookie>
     */
    public static HttpCookie getCookie(ServerWebExchange exchange, String name) {
        return getCookies(exchange).getFirst(name);
    }

    /**
     * 读取 {@link ServerWebExchange} 某个名称对应的值
     *
     * @param exchange {@link ServerWebExchange}
     * @param name     Cookie Key
     * @return cookie 值
     */
    public static String getCookieValue(ServerWebExchange exchange, String name) {
        HttpCookie cookie = getCookie(exchange, name);
        return ObjectUtils.isNotEmpty(cookie) ? cookie.getValue() : null;
    }

    /**
     * 获取 COOKIE 头请求头内容
     *
     * @param exchange {@link ServerWebExchange}
     * @return COOKIE 请求头内容
     */
    public static String getCookieFromHeader(ServerWebExchange exchange) {
        return HeaderUtils.getFirstHeader(exchange, HttpHeaders.COOKIE);
    }

    /**
     * 获取 {@link ServerWebExchange} Cookie 请求头中，某个属性的值
     *
     * @param exchange {@link ServerWebExchange}
     * @param name     Cookie中的属性名
     * @return 如果 Cookie 存在属性名就返回对应的值，如果不存在则返回null
     */
    public static String getCookieValueFromHeader(ServerWebExchange exchange, String name) {
        String cookie = getCookieFromHeader(exchange);
        return get(cookie, name);
    }

    /**
     * 从 {@link ServerWebExchange} Cookie 请求头中，找到给定任意给定属性的值
     *
     * @param exchange {@link ServerWebExchange}
     * @param name     Cookie中的属性名
     * @return 如果 Cookie 存在属性名就返回对应的值，如果不存在则返回null
     */
    public static String getAnyCookieValueFromHeader(ServerWebExchange exchange, String... name) {
        String cookie = getCookieFromHeader(exchange);
        return getAny(cookie, name);
    }
}
