/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.web.core.reactive.utils;

import io.netty.buffer.ByteBufAllocator;
import org.dromara.hutool.core.text.StrUtil;
import org.dromara.hutool.core.util.ByteUtil;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferUtils;
import org.springframework.core.io.buffer.NettyDataBufferFactory;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.util.concurrent.atomic.AtomicReference;

/**
 * <p>Description: WebFlux 请求体处理通用方法 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/4/17 21:39
 */
public class RequestBodyUtils {

    /**
     * 字节数组转 {@link DataBuffer}
     *
     * @param bytes 字节数组
     * @return {@link DataBuffer}
     */
    public static DataBuffer toDataBuffer(byte[] bytes) {
        NettyDataBufferFactory nettyDataBufferFactory = new NettyDataBufferFactory(ByteBufAllocator.DEFAULT);
        DataBuffer buffer = nettyDataBufferFactory.allocateBuffer(bytes.length);
        buffer.write(bytes);
        return buffer;
    }

    /**
     * 字符串转 {@link DataBuffer}
     *
     * @param value 字符串数据
     * @return {@link DataBuffer}
     */
    public static DataBuffer toDataBuffer(String value) {
        byte[] bytes = ByteUtil.toUtf8Bytes(value);
        return toDataBuffer(bytes);
    }

    /**
     * 读取 {@link DataBuffer} 并转为 byte 数组
     *
     * @param dataBuffer {@link DataBuffer}
     * @return byte 数组
     */
    public static byte[] toBytes(DataBuffer dataBuffer) {
        byte[] bytes = new byte[dataBuffer.readableByteCount()];
        dataBuffer.read(bytes);
        // 释放堆外内存
        DataBufferUtils.release(dataBuffer);
        return bytes;
    }

    /**
     * 读取 {@link DataBuffer} 并转为 String
     *
     * @param dataBuffer {@link DataBuffer}
     * @return string
     */
    public static String toString(DataBuffer dataBuffer) {
        byte[] bytes = toBytes(dataBuffer);
        return StrUtil.utf8Str(bytes);
    }

    /**
     * 将字符串类型的 RequestBody 转换为 Flux<DataBuffer> 类型的 RequestBody
     *
     * @param body RequestBody {@link String}
     * @return RequestBody Flux<DataBuffer>
     */
    public static Flux<DataBuffer> stringToFlux(String body) {
        return Flux.just(toDataBuffer(body));
    }

    /**
     * 读取 WebFlux POST 类型请求的 RequestBody。并将其转换为字符串
     *
     * @return 请求体
     */
    public static String fluxToString(Flux<DataBuffer> body) {
        AtomicReference<String> atomicReference = new AtomicReference<>();
        body.subscribe(dataBuffer -> atomicReference.set(toString(dataBuffer)));
        return atomicReference.get();
    }

    /**
     * 缓存请求 RequestBody
     *
     * @param exchange 请求 {@link ServerWebExchange}
     * @param bytes    缓存内容
     * @return 缓存数据
     */
    public static Flux<DataBuffer> cached(ServerWebExchange exchange, byte[] bytes) {
        return Flux.defer(() -> {
            DataBuffer buffer = exchange.getResponse().bufferFactory().wrap(bytes);
            DataBufferUtils.retain(buffer);
            return Mono.just(buffer);
        });
    }

    /**
     * 从请求中获取请求体并转为字符串
     *
     * @param exchange {@link ServerWebExchange}
     * @return 字符串类型请求 Body
     */
    public static String getBodyString(ServerWebExchange exchange) {
        return getBodyString(exchange.getRequest());
    }

    /**
     * 从请求中获取请求体并转为字符串
     *
     * @param request {@link ServerHttpRequest}
     * @return 字符串类型请求 Body
     */
    public static String getBodyString(ServerHttpRequest request) {
        return fluxToString(request.getBody());
    }
}
