/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Cloud.
 *
 * Herodotus Cloud is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Cloud is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.web.core.reactive.utils;

import cn.herodotus.stirrup.web.core.constants.WebConstants;
import org.springframework.web.server.ServerWebExchange;
import org.springframework.web.server.WebSession;
import reactor.core.publisher.Mono;

/**
 * <p>Description: WebFlux Session 工具类 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/2/5 20:47
 */
public class SessionUtils {

    /**
     * 读取 {@link ServerWebExchange} 中的 {@link WebSession}
     *
     * @param exchange {@link ServerWebExchange}
     * @return Mono<WebSession>
     */
    public static Mono<WebSession> getSession(ServerWebExchange exchange) {
        return exchange.getSession();
    }

    /**
     * 获取 Session ID
     *
     * @param exchange {@link ServerWebExchange}
     * @return session ID 或者 null
     */
    public static String getSessionIdFromHeader(ServerWebExchange exchange) {
        return CookieUtils.getAnyCookieValueFromHeader(exchange, WebConstants.SESSION_IDS);
    }


    /**
     * 解析 Session ID
     * <p>
     * 如果请求中有 X_HERODOTUS_SESSION_ID 头，那么则返回 SessionId，意味着前后端加密有效。
     * 这种处理方式，主要解决在没有使用系统 Session 的环境下，单独调用接口特别是测试接口时，提示 Session 过期的问题。
     *
     * @param exchange {@link ServerWebExchange}
     * @return session ID 或者 null
     */
    public static Mono<String> analyseSessionId(ServerWebExchange exchange) {
        Mono<WebSession> webSessionMono = getSession(exchange);
        return webSessionMono
                .map(WebSession::getId)
                .switchIfEmpty(Mono.defer(() -> Mono.justOrEmpty(getSessionIdFromHeader(exchange))
                        .switchIfEmpty(Mono.defer(() -> Mono.justOrEmpty(HeaderUtils.getHerodotusTenantId(exchange))))));
    }
}
