/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.web.core.reactive.utils;

import cn.herodotus.stirrup.web.core.reactive.definition.RequestBodyModifier;
import org.apache.commons.lang3.ObjectUtils;
import org.dromara.hutool.core.util.ByteUtil;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.http.server.reactive.ServerHttpRequestDecorator;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

import java.lang.reflect.Array;
import java.net.URI;
import java.util.Optional;
import java.util.function.Function;

/**
 * <p>Description: WebFlux Exchange 工具类 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/2/5 20:51
 */
public class ExchangeUtils {

    /**
     * 读取 {@link ServerHttpRequest} 中的请求 URI
     *
     * @param exchange {@link ServerWebExchange}
     * @return 请求路径
     */
    public static URI getRequestURI(ServerWebExchange exchange) {
        return exchange.getRequest().getURI();
    }

    /**
     * 读取 {@link ServerWebExchange} 中的请求路径
     *
     * @param exchange {@link ServerWebExchange}
     * @return 请求路径
     */
    public static String getRequestPath(ServerWebExchange exchange) {
        return exchange.getRequest().getURI().getPath();
    }

    /**
     * 读取 {@link ServerWebExchange} 中的请求方法
     *
     * @param exchange {@link ServerWebExchange}
     * @return 请求方法 {@link HttpMethod}
     */
    public static HttpMethod getHttpMethod(ServerWebExchange exchange) {
        return exchange.getRequest().getMethod();
    }

    /**
     * 重写 WebFlux Request，设置新的 Headers 和 RequestBody
     *
     * @param request 请求 {@link ServerHttpRequest}
     * @param headers 新的请求头 {@link HttpHeaders}
     * @param body    新的请求体 {@link Flux}
     * @return 重写后的请求 {@link ServerWebExchange}
     */
    public static ServerHttpRequest decorateRequest(ServerHttpRequest request, HttpHeaders headers, Flux<DataBuffer> body) {
        return new ServerHttpRequestDecorator(request) {
            @Override
            public HttpHeaders getHeaders() {
                return headers;
            }

            @Override
            public Flux<DataBuffer> getBody() {
                return body;
            }
        };
    }

    /**
     * 根据 URI 重新构建 WebFlux 请求
     *
     * @param exchange {@link ServerWebExchange}
     * @return 新的请求 {@link ServerHttpRequest}
     */
    public static ServerHttpRequest createNewRequest(ServerWebExchange exchange) {
        return createNewRequest(exchange, null);
    }

    /**
     * 根据 URI 重新构建 WebFlux 请求
     *
     * @param exchange {@link ServerWebExchange}
     * @param uri      新的请求地址 {@link URI}
     * @return 新的请求 {@link ServerHttpRequest}
     */
    public static ServerHttpRequest createNewRequest(ServerWebExchange exchange, URI uri) {
        if (ObjectUtils.isEmpty(uri)) {
            uri = getRequestURI(exchange);
        }
        return exchange.getRequest().mutate().uri(uri).build();
    }

    /**
     * 重新构建 {@link ServerWebExchange}
     *
     * @param exchange {@link ServerWebExchange}
     * @param request  新请求 {@link ServerHttpRequest}
     * @return 新的 exchange  {@link ServerWebExchange}
     */
    public static ServerWebExchange createNewExchange(ServerWebExchange exchange, ServerHttpRequest request) {
        return exchange.mutate().request(request).build();
    }

    /**
     * 重新构建 {@link ServerWebExchange}
     *
     * @param exchange {@link ServerWebExchange}
     * @param uri      新的请求地址 {@link URI}
     * @return 新的 exchange  {@link ServerWebExchange}
     */
    public static ServerWebExchange createNewExchange(ServerWebExchange exchange, URI uri) {
        ServerHttpRequest request = createNewRequest(exchange, uri);
        return createNewExchange(exchange, request);
    }

    /**
     * 将修改后的请求体封装为新 Exchange 的统一方法
     *
     * @param exchange    {@link ServerWebExchange}
     * @param requestBody 字符串类型的请求体
     * @return 新的 exchange  {@link ServerWebExchange}
     */
    public static ServerWebExchange createNewExchange(ServerWebExchange exchange, String requestBody) {
        // 重新构建一个请求
        ServerHttpRequest request = ExchangeUtils.createNewRequest(exchange);
        // 字符串类型的 RequestBody 转成 Flux<DataBuffer>类型
        Flux<DataBuffer> bodyFlux = RequestBodyUtils.stringToFlux(requestBody);
        // 请求体内容可能存在变化，需要根据新的请求内容，重新设置 CONTENT_LENGTH
        HttpHeaders headers = HeaderUtils.resetContentLength(exchange, Array.getLength(ByteUtil.toUtf8Bytes(requestBody)));
        // 用新的请求头和请求体重新包装请求
        request = ExchangeUtils.decorateRequest(request, headers, bodyFlux);
        // 返回新的 Exchange
        return ExchangeUtils.createNewExchange(exchange, request);
    }

    /**
     * 重写 WebFlux Request Body
     *
     * @param exchange 请求 {@link ServerWebExchange}
     * @param headers  新的请求头 {@link HttpHeaders}
     * @param body     新的请求体 {@link Flux}
     * @return 重写后的请求 {@link ServerWebExchange}
     */
    public static ServerWebExchange createNewExchange(ServerWebExchange exchange, HttpHeaders headers, Flux<DataBuffer> body) {
        ServerHttpRequest mutatedRequest = decorateRequest(exchange.getRequest(), headers, body);
        return createNewExchange(exchange, mutatedRequest);
    }

    /**
     * 修改 {@link ServerWebExchange} 请求体
     * <p>
     * WebFlux 中对 POST 类型请求 RequestBody 内容的修改，其实很套路化。XSS 防护和 SQL 注入都需要对 RequestBody 进行修改，除了核心检测处理逻辑之外，整体逻辑大同小异。
     * 所以提取了一个公共的方法，实现 WebFlux XSS 防护和 SQL 注入 RequestBody 内容修改的简化。其它场景也可以使用。
     *
     * @param exchange 请求 {@link ServerWebExchange}
     * @param modifier 请求体修改器 {@link RequestBodyModifier}
     * @param onFinish 完成操作
     * @param onError  错误操作
     * @return Mono<Void>
     */
    public static Mono<Void> modify(ServerWebExchange exchange, RequestBodyModifier modifier, Function<ServerWebExchange, Mono<Void>> onFinish, Function<ServerWebExchange, Mono<Void>> onError) {
        return DataBufferUtils.join(exchange.getRequest().getBody())
                .switchIfEmpty(onFinish.apply(exchange).then(Mono.empty()))
                .map(RequestBodyUtils::toString)
                .map(originalBody -> Optional.ofNullable(modifier.apply(exchange, originalBody)))
                .flatMap(newBody -> newBody.isPresent() ? onFinish.apply(createNewExchange(exchange, newBody.get())) : onError.apply(exchange));
    }

    /**
     * 判断请求体中的数据是否为 application/json
     *
     * @param contentType Content Type
     * @return true 请求体数据类型为 application/json，false 请求体数据类型不是 application/json
     */
    public static boolean isJson(String contentType) {
        return MediaType.APPLICATION_JSON_VALUE.equalsIgnoreCase(contentType) || MediaType.APPLICATION_JSON_UTF8_VALUE.equalsIgnoreCase(contentType);
    }

    /**
     * 判断请求体中的数据是否为 JSON
     *
     * @param exchange 请求对象 {@link ServerWebExchange}
     * @return true 请求体数据类型为 json，false 请求体数据类型不是 json
     */
    public static boolean isJson(ServerWebExchange exchange) {
        String contentType = HeaderUtils.getContentType(exchange);
        return isJson(contentType);
    }

    /**
     * 判断请求体中的数据是否为 application/x-www-form-urlencoded
     *
     * @param contentType Content Type
     * @return true 请求体数据类型为 application/x-www-form-urlencoded，false 请求体数据类型不是 application/x-www-form-urlencoded
     */
    public static boolean isFormUrlencoded(String contentType) {
        return MediaType.APPLICATION_FORM_URLENCODED_VALUE.equalsIgnoreCase(contentType);
    }

    /**
     * 判断请求体中的数据是否为 application/x-www-form-urlencoded
     *
     * @param exchange 请求对象 {@link ServerWebExchange}
     * @return true 请求体数据类型为 application/x-www-form-urlencoded，false 请求体数据类型不是 application/x-www-form-urlencoded
     */
    public static boolean isFormUrlencoded(ServerWebExchange exchange) {
        String contentType = HeaderUtils.getContentType(exchange);
        return isJson(contentType);
    }

    /**
     * 判断是否为 GET 类型的请求，这里的 GET 类型是指通过 query 方式传递参数的请求类型。GET 类型请求包括 GET 和 DELETE
     *
     * @param method 请求类型 {@link HttpMethod}
     * @return true 是 Get 类型，false 不是 Get 类型
     */
    public static boolean isGetTypeRequest(HttpMethod method) {
        return method == HttpMethod.GET || method == HttpMethod.DELETE;
    }

    /**
     * 判断是否为 GET 类型的请求，这里的 GET 类型是指通过 query 方式传递参数的请求类型。GET 类型请求包括 GET 和 DELETE
     *
     * @param exchange 请求对象 {@link ServerWebExchange}
     * @return true 是 Get 类型，false 不是 Get 类型
     */
    public static boolean isGetTypeRequest(ServerWebExchange exchange) {
        HttpMethod method = getHttpMethod(exchange);
        return isGetTypeRequest(method);
    }

    /**
     * 判断是否为 POST 类型的请求，这里的 POST 类型是指通过 RequestBody 方式传递参数的请求类型。POST 类型请求包括 POST 和 PUT
     *
     * @param method      请求类型 {@link HttpMethod}
     * @param contentType 内容类型
     * @return true 是 POST 类型，false 不是 POST 类型
     */
    public static Boolean isPostTypeRequest(HttpMethod method, String contentType) {
        return (method == HttpMethod.POST || method == HttpMethod.PUT) && (isFormUrlencoded(contentType) || isJson(contentType));
    }

    /**
     * 判断是否为 POST 类型的请求，这里的 POST 类型是指通过 RequestBody 方式传递参数的请求类型。POST 类型请求包括 POST 和 PUT
     *
     * @param exchange 请求对象 {@link ServerWebExchange}
     * @return true 是 POST 类型，false 不是 POST 类型
     */
    public static Boolean isPostTypeRequest(ServerWebExchange exchange) {
        HttpMethod method = getHttpMethod(exchange);
        String contentType = HeaderUtils.getContentType(exchange);
        return isPostTypeRequest(method, contentType);
    }
}
