/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2020-2030 郑庚伟 ZHENGGENGWEI (码匠君), <herodotus@aliyun.com> Licensed under the AGPL License
 *
 * This file is part of Herodotus Stirrup.
 *
 * Herodotus Stirrup is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Herodotus Stirrup is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.herodotus.vip>.
 */

package cn.herodotus.stirrup.web.core.reactive.utils;

import cn.herodotus.stirrup.core.definition.constants.HerodotusHeaders;
import com.google.common.collect.ImmutableList;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.springframework.http.HttpHeaders;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.web.server.ServerWebExchange;

import java.util.List;

/**
 * <p>Description: 响应式请求头工具类 </p>
 *
 * @author : gengwei.zheng
 * @date : 2024/2/3 14:03
 */
public class HeaderUtils {

    /**
     * 读取 {@link ServerWebExchange} 中的 全部请求头信息
     *
     * @param exchange {@link ServerWebExchange}
     * @return 头信息 {@link HttpHeaders}
     */
    public static HttpHeaders getHeaders(ServerWebExchange exchange) {
        return exchange.getRequest().getHeaders();
    }

    /**
     * 获取某个请求头的值
     *
     * @param exchange {@link ServerWebExchange}
     * @param name     名称
     * @return 头信息值 List<String>
     */
    public static List<String> getHeader(ServerWebExchange exchange, String name) {
        HttpHeaders httpHeaders = getHeaders(exchange);
        if (MapUtils.isNotEmpty(httpHeaders)) {
            return httpHeaders.get(name);
        }
        return ImmutableList.of();
    }

    /**
     * 获取第一个头信息
     *
     * @param exchange {@link ServerWebExchange}
     * @param name     名称
     * @return 头信息值
     */
    public static String getFirstHeader(ServerWebExchange exchange, String name) {
        List<String> header = getHeader(exchange, name);
        if (CollectionUtils.isNotEmpty(header)) {
            return header.getFirst();
        }

        return null;
    }

    /**
     * 获取自定义 X_HERODOTUS_SESSION_ID 请求头内容
     *
     * @param exchange {@link ServerWebExchange}
     * @return X_HERODOTUS_SESSION_ID 请求头内容
     */
    public static String getHerodotusSessionId(ServerWebExchange exchange) {
        return getFirstHeader(exchange, HerodotusHeaders.X_HERODOTUS_SESSION_ID);
    }

    /**
     * 获取自定义 X_HERODOTUS_TENANT_ID 请求头内容
     *
     * @param exchange {@link ServerWebExchange}
     * @return X_HERODOTUS_TENANT_ID 请求头内容
     */
    public static String getHerodotusTenantId(ServerWebExchange exchange) {
        return getFirstHeader(exchange, HerodotusHeaders.X_HERODOTUS_TENANT_ID);
    }

    /**
     * 获取自定义 X_HERODOTUS_FROM_IN 请求头内容
     *
     * @param exchange {@link ServerWebExchange}
     * @return X_HERODOTUS_FROM_IN 请求头内容
     */
    public static String getHerodotusFromIn(ServerWebExchange exchange) {
        return getFirstHeader(exchange, HerodotusHeaders.X_HERODOTUS_FROM_IN);
    }

    /**
     * 获取 AUTHORIZATION 请求头内容
     *
     * @param exchange {@link ServerWebExchange}
     * @return AUTHORIZATION 请求头内容
     */
    public static String getAuthorization(ServerWebExchange exchange) {
        return getFirstHeader(exchange, HttpHeaders.AUTHORIZATION);
    }

    /**
     * 获取 CONTENT_TYPE 请求头内容
     *
     * @param exchange {@link ServerWebExchange}
     * @return CONTENT_TYPE 请求头内容
     */
    public static String getContentType(ServerWebExchange exchange) {
        return getFirstHeader(exchange, HttpHeaders.CONTENT_TYPE);
    }

    /**
     * 获取 SEC_WEBSOCKET_PROTOCOL 请求头内容
     *
     * @param exchange {@link ServerWebExchange}
     * @return AUTHORIZATION 请求头内容
     */
    public static String getWebsocketProtocol(ServerWebExchange exchange) {
        return getFirstHeader(exchange, com.google.common.net.HttpHeaders.SEC_WEBSOCKET_PROTOCOL);
    }

    /**
     * 设置 Header
     *
     * @param headers     请求/响应头容器
     * @param headerName  头名称
     * @param headerValue 头值
     */
    public static void setHeaders(HttpHeaders headers, String headerName, String headerValue) {
        headers.set(headerName, headerValue);
    }

    /**
     * 重新创建一个新的 Header，重新设置请求体内容大小。
     *
     * @param exchange      {@link ServerWebExchange}
     * @param contentLength 新的内容长度
     * @return 新的 Headers {@link HttpHeaders}
     */
    public static HttpHeaders resetContentLength(ServerWebExchange exchange, int contentLength) {
        HttpHeaders headers = new HttpHeaders();
        headers.putAll(HeaderUtils.getHeaders(exchange));

        // 由于修改了传递参数，需要重新设置CONTENT_LENGTH，长度是字节长度，不是字符串长度
        headers.remove(HttpHeaders.CONTENT_LENGTH);
        headers.setContentLength(contentLength);
        return headers;
    }

    /**
     * 添加响应的 Content-Type
     *
     * @param response    响应对象
     * @param contentType 值
     */
    public static void addResponseContentType(ServerHttpResponse response, String contentType) {
        response.getHeaders().add(HttpHeaders.CONTENT_TYPE, contentType);
    }
}
