package cn.hollycloud.iplatform.common.utils;

import cn.hollycloud.iplatform.common.annotation.*;
import cn.hollycloud.iplatform.common.bean.TreeBean;
import cn.hollycloud.iplatform.common.exception.ServiceFailException;
import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ReflectUtil;
import com.alibaba.fastjson.JSONArray;

import java.lang.reflect.Field;
import java.util.*;

/**
 * Description:
 *
 * @author: Cloud
 * Date: 2019-10-30
 * Time: 16:02
 */
public class TreeUtils {
    private final static List DEFAULT_ROOT_IDS = Arrays.asList(null, 0, 0L);

    private static <T> List<TreeBean<T>> parseTree(List<T> list, Field idField, Field parentIdField, Field displayField, List rootIds) {
        if (CollectionUtil.isEmpty(rootIds)) {
            rootIds = DEFAULT_ROOT_IDS;
        }
        List<TreeBean<T>> tree = new ArrayList<>();
        List<TreeBean> treeBeans = handleTree(list, idField, parentIdField, displayField);
        for (TreeBean treeBean : treeBeans) {
            if (rootIds.contains(treeBean.getParentId())) {
                tree.add(treeBean);
            }
        }
        return tree;
    }

    //把列表解析成树结构
    public static <T> List<TreeBean<T>> parseTree(List<T> list, Class<T> clazz) {
        return parseTree(list, clazz, null);
    }

    //把列表解析成树结构,并指定根节点id
    public static <T> List<TreeBean<T>> parseTree(List<T> list, Class<T> clazz, List rootIds) {
        Field[] fields = ReflectUtil.getFields(clazz);
        Field idField = null;
        Field parentIdField = null;
        Field displayField = null;
        for (Field field : fields) {
            if (field.getAnnotation(TreeId.class) != null) {
                field.setAccessible(true);
                idField = field;
            } else if (field.getAnnotation(TreeParentId.class) != null) {
                field.setAccessible(true);
                parentIdField = field;
            } else if (field.getAnnotation(TreeName.class) != null) {
                field.setAccessible(true);
                displayField = field;
            }
        }
        if (idField == null) {
            throw new ServiceFailException("没有找到TreeId注解");
        }
        if (parentIdField == null) {
            throw new ServiceFailException("没有找到TreeParentId注解");
        }
        if (displayField == null) {
            throw new ServiceFailException("没有找到TreeName注解");
        }
        return parseTree(list, idField, parentIdField, displayField, rootIds);
    }

    // 组装树，返回扁平树，树下节点已挂载
    private static List<TreeBean> handleTree(List list, Field idField, Field parentIdField, Field displayField) {
        try {
            List<TreeBean> tree = new ArrayList<>();
            Map<Object, TreeBean> map = new HashMap();
            //把列表映射成id,obj形式
            for (int i = 0; i < list.size(); i++) {
                Object object = list.get(i);
                TreeBean treeBean = new TreeBean();
                Object id = idField.get(object);
                if (id == null) {
                    continue;
                }
                treeBean.setId(id);
                Object name = displayField.get(object);
                if (name == null) {
                    continue;
                }
                treeBean.setDisplayName(name);
                Object parentId = parentIdField.get(object);
                if (parentId != null) {
                    treeBean.setParentId(parentId);
                }
                treeBean.setNativeObject(object);
                treeBean.setDisplayName(name);
                map.put(id, treeBean);
                tree.add(treeBean);
            }

            for (int i = 0; i < tree.size(); i++) {
                TreeBean object = tree.get(i);
                Object parentId = object.getParentId();
                if (parentId != null) {
                    //父节点为null表示根节点
                    TreeBean parentNode = map.get(parentId);
                    if (parentNode != null) {
                        List<TreeBean> children = parentNode.getChildren();
                        children.add(object);
                        parentNode.setIsLeaf(false);
                    }
                }
            }
            return tree;
        } catch (Exception e) {
            throw new ServiceFailException("解析树出错", e);
        }
    }

    private static List<TreeBean> mountTree(List parentList, Field idField, Field parentIdField, Field displayField,
                                            List childList, Field childIdField, Field childParentIdField, Field childDisplayField, List rootIds) {
        try {
            if (CollectionUtil.isEmpty(rootIds)) {
                rootIds = DEFAULT_ROOT_IDS;
            }
            List<TreeBean> tree = new ArrayList<>();
            List<TreeBean> treeBeans = handleTree(parentList, idField, parentIdField, displayField);
            Map<Object, TreeBean> map = new HashMap();
            //把列表映射成id,obj形式
            for (int i = 0; i < treeBeans.size(); i++) {
                TreeBean treeBean = treeBeans.get(i);
                if (rootIds.contains(treeBean.getParentId())) {
                    tree.add(treeBean);
                }
                map.put(treeBean.getId(), treeBean);
            }

            for (int i = 0; i < childList.size(); i++) {
                Object childNode = childList.get(i);
                TreeBean treeBean = new TreeBean();
                treeBean.setType(1);
                treeBean.setIsLeaf(true);
                Object id = childIdField.get(childNode);
                if (id == null) {
                    continue;
                }
                treeBean.setId(id);
                Object name = childDisplayField.get(childNode);
                if (name == null) {
                    continue;
                }
                treeBean.setDisplayName(name);
                treeBean.setNativeObject(childNode);
                Object parentId = childParentIdField.get(childNode);
                if (parentId == null) {
                    tree.add(treeBean);
                } else {
                    treeBean.setParentId(parentId);
                    TreeBean parentNode = map.get(parentId);
                    if (parentNode == null) {
                        continue;
                    }
                    parentNode.setIsLeaf(false);
                    List<TreeBean> children = parentNode.getChildren();
                    children.add(treeBean);
                }
            }
            return tree;
        } catch (Exception e) {
            throw new ServiceFailException(e);
        }
    }

    public static List<TreeBean> mountTree(List parentList, Class parentClass, List childList, Class childClass) {
        return mountTree(parentList, parentClass, childList, childClass, null);
    }

    public static List<TreeBean> mountTree(List parentList, Class parentClass, List childList, Class childClass, List rootIds) {
        Field[] fields = ReflectUtil.getFields(parentClass);
        Field idField = null;
        Field parentIdField = null;
        Field displayField = null;

        for (Field field : fields) {
            if (field.getAnnotation(TreeId.class) != null) {
                field.setAccessible(true);
                idField = field;
            } else if (field.getAnnotation(TreeParentId.class) != null) {
                field.setAccessible(true);
                parentIdField = field;
            } else if (field.getAnnotation(TreeName.class) != null) {
                field.setAccessible(true);
                displayField = field;
            }
        }
        if (idField == null) {
            throw new ServiceFailException("没有找到TreeId注解");
        }
        if (parentIdField == null) {
            throw new ServiceFailException("没有找到TreeParentId注解");
        }
        if (displayField == null) {
            throw new ServiceFailException("没有找到TreeName注解");
        }

        fields = ReflectUtil.getFields(childClass);
        Field childIdField = null;
        Field childParentIdField = null;
        Field childDisplayField = null;
        for (Field field : fields) {
            if (field.getAnnotation(TreeId.class) != null) {
                field.setAccessible(true);
                childIdField = field;
            } else if (field.getAnnotation(TreeParentId.class) != null) {
                field.setAccessible(true);
                childParentIdField = field;
            } else if (field.getAnnotation(TreeName.class) != null) {
                field.setAccessible(true);
                childDisplayField = field;
            }
        }
        if (childIdField == null) {
            throw new ServiceFailException("没有找到子节点TreeId注解");
        }
        if (childParentIdField == null) {
            throw new ServiceFailException("没有找到子节点TreeParentId注解");
        }
        if (childDisplayField == null) {
            throw new ServiceFailException("没有找到子节点TreeName注解");
        }
        return mountTree(parentList, idField, parentIdField, displayField,
                childList, childIdField, childParentIdField, childDisplayField, rootIds);
    }

    public static <T> List<T> parseStandardTree(List<T> list, Class<T> clazz) {
        return parseStandardTree(list, clazz, null);
    }

    public static <T> List<T> parseStandardTree(List<T> list, Class<T> clazz, List rootIds) {
        Field[] fields = ReflectUtil.getFields(clazz);
        Field idField = null;
        Field parentIdField = null;
        Field childrenField = null;
        Field leafField = null;
        for (Field field : fields) {
            if (field.getAnnotation(TreeId.class) != null) {
                field.setAccessible(true);
                idField = field;
            } else if (field.getAnnotation(TreeParentId.class) != null) {
                field.setAccessible(true);
                parentIdField = field;
            } else if (field.getAnnotation(TreeChildren.class) != null) {
                field.setAccessible(true);
                //判断field类型是否是collection子类
                if (!Collection.class.isAssignableFrom(field.getType())) {
                    throw new ServiceFailException("TreeChildren对应类型必须是容器类");
                }
                childrenField = field;
            } else if (field.getAnnotation(TreeLeaf.class) != null) {
                if (field.getType() != Boolean.class) {
                    throw new ServiceFailException("TreeLeaf对应类型必须是Boolean");
                }
                field.setAccessible(true);
                leafField = field;
            }
        }
        if (idField == null) {
            throw new ServiceFailException("没有找到TreeId注解");
        }
        if (parentIdField == null) {
            throw new ServiceFailException("没有找到TreeParentId注解");
        }
        if (childrenField == null) {
            throw new ServiceFailException("没有找到TreeChildren注解");
        }
        return parseStandardTree(list, idField, parentIdField, childrenField, leafField, rootIds);
    }

    private static <T> List<T> parseStandardTree(List<T> list, Field idField, Field parentIdField, Field childrenField, Field leafField, List rootIds) {
        try {
            if (CollectionUtil.isEmpty(rootIds)) {
                rootIds = DEFAULT_ROOT_IDS;
            }
            List<T> tree = new ArrayList<>();
            List<T> treeBeans = handleStandardTree(list, idField, parentIdField, childrenField, leafField);
            for (T treeBean : treeBeans) {
                if (rootIds.contains(parentIdField.get(treeBean))) {
                    tree.add(treeBean);
                }
            }
            return tree;
        } catch (Exception e) {
            throw new ServiceFailException("解析树出错", e);
        }
    }

    private static <T> List<T> handleStandardTree(List<T> list, Field idField, Field parentIdField, Field childrenField, Field leafField) throws IllegalAccessException {
        List<T> tree = new ArrayList<>();
        Map<Object, T> map = new HashMap();
        //把列表映射成id,obj形式
        for (int i = 0; i < list.size(); i++) {
            T object = list.get(i);
            if (leafField != null) {
                leafField.set(object, true);
            }
            Object id = idField.get(object);
            if (id == null) {
                continue;
            }

            map.put(id, object);
            tree.add(object);
        }
        //
        for (int i = 0; i < tree.size(); i++) {
            T object = tree.get(i);
            Object parentId = parentIdField.get(object);
            if (parentId != null) {
                //父节点为null表示根节点
                T parentNode = map.get(parentId);
                if (parentNode != null) {
                    Collection children = (Collection) childrenField.get(parentNode);
                    if (children == null) {
                        children = new JSONArray();
                        childrenField.set(parentNode, children);
                    }
                    children.add(object);
                    if (leafField != null) {
                        leafField.set(parentNode, false);
                    }
                }
            }
        }
        return tree;
    }
}
