package cn.huangxulin.githook.core;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.CommandLineRunner;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.StreamUtils;

import java.io.*;
import java.net.URL;
import java.util.Objects;

/**
 * 核心处理服务
 *
 * @author hxulin
 */
@Slf4j
public class GitHookService implements CommandLineRunner {

    /**
     * 编译生成的目标文件路径的后缀匹配
     */
    private static final String CLASS_PATH_SUFFIX = "/target/classes/";

    @Override
    public void run(String... args) {
        ClassPathResource resource = new ClassPathResource("git-hooks");
        if (!resource.exists()) {
            log.warn("git hook script not found in classpath/git-hooks");
            return;
        }
        File[] hookFiles = null;
        try {
            File hookDir = resource.getFile();
            if (hookDir.exists() && hookDir.isDirectory()) {
                hookFiles = hookDir.listFiles();
            }
        } catch (Exception ignored) {
            log.warn("git hook script not found in classpath/git-hooks");
            return;
        }
        if (hookFiles == null || hookFiles.length == 0) {
            log.warn("git hook script not found in classpath/git-hooks");
            return;
        }
        URL url = Thread.currentThread().getContextClassLoader().getResource("");
        String classPath = Objects.requireNonNull(url).getPath();
        if (!classPath.endsWith(CLASS_PATH_SUFFIX)) {
            log.warn("git hook classpath check failed");
            return;
        }
        // 截掉路径末尾 "target/classes/" 15个字符
        File hookDir = getHookDir(classPath.substring(0, classPath.length() - 15));
        if (hookDir == null) {
            log.warn("git hook profile storage location check failed");
            return;
        }
        for (File hookFile : hookFiles) {
            String fileName = hookFile.getName();
            File targetFile = new File(hookDir, fileName);
            try (
                    InputStream in = new FileInputStream(hookFile);
                    OutputStream out = new FileOutputStream(targetFile)
            ) {
                StreamUtils.copy(in, out);
            } catch (IOException ignored) {
                log.error("git hook profile init failed");
            }
            // 给脚本文件设置可执行权限
            if (targetFile.setExecutable(true, false)) {
                log.info("git hook script: {} executable permission cannot be set", targetFile.getName());
            }
        }
    }

    /**
     * 递归查找 hook dir 路径
     *
     * @param workDir 工作路径
     * @return hook dir
     */
    private File getHookDir(String workDir) {
        if (workDir == null) {
            return null;
        }
        File hookDir = new File(workDir, ".git/hooks");
        if (!hookDir.exists() || !hookDir.isDirectory()) {
            File targetDir = new File(workDir);
            if (targetDir.exists()) {
                return getHookDir(targetDir.getParent());
            }
            return null;
        }
        return hookDir;
    }
}
