package cn.keepbx.jpom.plugins;

import cn.hutool.core.convert.Convert;
import cn.hutool.core.util.ClassUtil;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;

import java.util.HashMap;
import java.util.Map;

/**
 * 插件模块接口
 *
 * @author bwcx_jzy
 * @since 2021/12/22
 */
public interface IPlugin extends AutoCloseable {

    /**
     * 数据目录 key
     */
    String DATE_PATH_KEY = "JPOM_DATE_PATH";
    /**
     * JPOM 版本
     */
    String JPOM_VERSION_KEY = "JPOM_VERSION";


    /**
     * 执行插件方法
     *
     * @param main      拦截到到对象
     * @param parameter 执行方法传人的参数
     * @return 返回值
     * @throws Exception 异常
     */
    Object execute(Object main, Map<String, Object> parameter) throws Exception;

    /**
     * 执行插件方法
     *
     * @param main       主参数
     * @param parameters 其他参数
     * @return 结果
     * @throws Exception 异常
     */
    default Object execute(Object main, Object... parameters) throws Exception {
        // 处理参数
        int length = parameters.length;
        Map<String, Object> map = new HashMap<>(length / 2);
        for (int i = 0; i < length; i += 2) {
            map.put(parameters[i].toString(), parameters[i + 1]);
        }
        return this.execute(main, map);
    }

    /**
     * 执行插件方法
     *
     * @param main       拦截到到对象
     * @param parameters 其他参数
     * @param <T>        泛型
     * @param cls        返回值类型
     * @return 返回值
     * @throws Exception 异常
     */
    default <T> T execute(Object main, Class<T> cls, Object... parameters) throws Exception {
        Object execute = this.execute(main, parameters);
        return this.convertResult(execute, cls);
    }

    /**
     * 执行插件方法
     *
     * @param main      拦截到到对象
     * @param parameter 执行方法传人的参数
     * @param <T>       泛型
     * @param cls       返回值类型
     * @return 返回值
     * @throws Exception 异常
     */
    default <T> T execute(Object main, Map<String, Object> parameter, Class<T> cls) throws Exception {
        Object execute = this.execute(main, parameter);
        return this.convertResult(execute, cls);
    }

    /**
     * 转换结果
     *
     * @param execute 结果
     * @param cls     返回值类型
     * @param <T>     泛型
     * @return 返回值类型
     */
    @SuppressWarnings("unchecked")
    default <T> T convertResult(Object execute, Class<T> cls) {
        if (execute == null) {
            return null;
        }
        Class<?> aClass = execute.getClass();
        if (ClassUtil.isSimpleValueType(aClass)) {
            return (T) Convert.convert(aClass, execute);
        }
        // json 数据
        if (execute instanceof JSONObject) {
            JSONObject jsonObject = (JSONObject) execute;
            return jsonObject.to(cls);
        }
        return (T) execute;
    }

    /**
     * 系统关闭，插件资源释放
     *
     * @throws Exception 异常
     */
    @Override
    default void close() throws Exception {
    }
}
