package cn.keepbx.jpom.model;

import cn.keepbx.jpom.BaseJsonMessage;
import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONFactory;
import com.alibaba.fastjson2.JSONObject;
import com.alibaba.fastjson2.JSONWriter;
import com.alibaba.fastjson2.writer.ObjectWriterImplToString;
import com.alibaba.fastjson2.writer.ObjectWriterProvider;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;

import java.math.BigInteger;

/**
 * 通用的json 数据格式
 *
 * @author bwcx_jzy
 * @since 2017/2/6.
 * @since 1.0
 */
@EqualsAndHashCode(callSuper = true)
@Data
@NoArgsConstructor
public class JsonMessage<T> extends BaseJsonMessage<T> {

    static {
        ObjectWriterProvider writerProvider = JSONFactory.getDefaultObjectWriterProvider();
        // long 类型自动转String
        writerProvider.register(Long.class, ObjectWriterImplToString.INSTANCE);
        writerProvider.register(long.class, ObjectWriterImplToString.INSTANCE);
        writerProvider.register(BigInteger.class, ObjectWriterImplToString.INSTANCE);
        writerProvider.register(Long.TYPE, ObjectWriterImplToString.INSTANCE);
        //
        JSONFactory.setUseJacksonAnnotation(false);
        // 枚举对象使用 枚举名称
        JSON.config(JSONWriter.Feature.WriteEnumsUsingName);
    }


    public JsonMessage(int code, String msg, T data) {
        super(code, msg, data);
    }

    public JsonMessage(int code, String msg) {
        this(code, msg, null);
    }

    /**
     * 将data 转换为对应实体
     *
     * @param tClass 类
     * @param <E>    泛型
     * @return Object
     */
    public <E> E getData(Class<E> tClass) {
        return JSON.to(tClass, this.data);
    }

    /**
     * @return json
     * @author bwcx_jzy
     */
    @Override
    public String toString() {
        return JSONObject.toJSONString(this);
    }

    public JSONObject toJson() {
        return (JSONObject) JSON.toJSON(this);
    }

    public static JSONObject toJson(int code, String msg) {
        return toJson(code, msg, null);
    }

    /**
     * 获取 json 对象
     *
     * @param code 状态码
     * @param msg  消息
     * @param data 数据
     * @param <T>  范型
     * @return json
     */
    public static <T> JSONObject toJson(int code, String msg, T data) {
        JsonMessage<T> jsonMessage = new JsonMessage<>(code, msg, data);
        return jsonMessage.toJson();
    }

    /**
     * @param code code
     * @param msg  msg
     * @return json
     * @author bwcx_jzy
     */
    public static String getString(int code, String msg) {
        return getString(code, msg, null);
    }

    /**
     * @param code code
     * @param msg  msg
     * @param data data
     * @return json
     * @author bwcx_jzy
     */
    public static String getString(int code, String msg, Object data) {
        return toJson(code, msg, data).toString();
    }
}
