#set( $symbol_pound = '#' )
#set( $symbol_dollar = '$' )
#set( $symbol_escape = '\' )
package ${package}.queue.impl;

import cn.hutool.core.lang.Assert;
import ${package}.queue.DelayedQueue;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RDelayedQueue;
import org.redisson.api.RQueue;
import org.redisson.api.RedissonClient;
import org.redisson.api.listener.ListAddListener;

import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * 基于redisson的延时队列实现
 *
 * @author yujiaxin
 */
@Slf4j
public class RedissonDelayedQueue<T> implements DelayedQueue<T> {
    private final String queueName;
    private final RDelayedQueue<T> internalQueue;
    private final CopyOnWriteArrayList<Consumer<T>> consumers = new CopyOnWriteArrayList<>();
    private final RQueue<T> consumeQueue;

    /**
     * 指定redisson队列名称的构造函数
     *
     * @param queueName      队列名称
     * @param redissonClient redisson客户端
     */
    public RedissonDelayedQueue(String queueName, RedissonClient redissonClient) {
        Assert.notBlank(queueName, "队列名不能为空");
        this.queueName = queueName;
        consumeQueue = redissonClient.getQueue(queueName);
        consumeQueue.addListener((ListAddListener) name -> {
            consume();
        });
        internalQueue = redissonClient.getDelayedQueue(consumeQueue);
    }

    @Override
    public void offer(T ele, long delay, TimeUnit timeUnit) {
        internalQueue.offer(ele, delay, timeUnit);
    }

    @Override
    public void offer(T ele, long delaySeconds) {
        internalQueue.offer(ele, delaySeconds, TimeUnit.SECONDS);
    }

    @Override
    public void register(Consumer<T> action) {
        consumers.add(action);
    }

    @Override
    public void consume() {
        T ele = consumeQueue.poll();
        while (ele != null) { // 循环消费，避免挤压消息
            for (Consumer<T> consumer : consumers) {
                consumer.accept(ele);
            }
            ele = consumeQueue.poll();
        }
    }

    @Override
    public boolean remove(T ele) {
        boolean ret = internalQueue.remove(ele);
        boolean ret2 = consumeQueue.remove(ele);
        return ret || ret2;
    }

    public String getQueueName() {
        return queueName;
    }
}
